<?php
require_once __DIR__ . '/Partie.php';
require_once __DIR__ . '/Vaisseau.php';

class IAJoueur {
    private $difficulte; // 'facile', 'moyen', 'difficile'
    
    public function __construct($difficulte = 'moyen') {
        $this->difficulte = $difficulte;
    }
    
    /**
     * Point d'entrée principal : l'IA joue son tour
     */
    public function jouerTour($partie, $idJoueur) {
        $vaisseau = ($idJoueur === 1) ? $partie->joueur1 : $partie->joueur2;
        $adversaire = ($idJoueur === 1) ? $partie->joueur2 : $partie->joueur1;
        
        // Si le vaisseau est paralysé, passer le tour
        if ($vaisseau->bloqueMouvement > 0) {
            $vaisseau->rechargerEnergie(3);
            $partie->nouveauMessage("L'IA passe son tour (Immobilisé).", 0);
            $partie->passerTour();
            return;
        }
        
        // Choisir et exécuter la meilleure action
        $action = $this->choisirMeilleureAction($partie, $vaisseau, $adversaire, $idJoueur);
        $this->executerAction($partie, $vaisseau, $adversaire, $action, $idJoueur);
    }
    
    /**
     * Choisir la meilleure action selon le niveau de difficulté
     */
    private function choisirMeilleureAction($partie, $vaisseau, $adversaire, $idJoueur) {
        switch ($this->difficulte) {
            case 'facile':
                return $this->choisirActionFacile($partie, $vaisseau, $adversaire, $idJoueur);
            case 'moyen':
                return $this->choisirActionMoyen($partie, $vaisseau, $adversaire, $idJoueur);
            case 'difficile':
                return $this->choisirActionDifficile($partie, $vaisseau, $adversaire, $idJoueur);
            default:
                return $this->choisirActionMoyen($partie, $vaisseau, $adversaire, $idJoueur);
        }
    }
    
    /**
     * IA FACILE : Actions aléatoires avec biais vers recharge
     */
    private function choisirActionFacile($partie, $vaisseau, $adversaire, $idJoueur) {
        // 70% du temps : recharge si énergie < 8
        if ($vaisseau->getEnergie() < 8 && rand(1, 100) <= 70) {
            return ['type' => 'attendre'];
        }
        
        // Sinon, choisir aléatoirement entre déplacement et autres actions
        $choix = rand(1, 100);
        
        if ($choix <= 60) {
            // Déplacement aléatoire
            $casesAccessibles = $vaisseau->getCasesAccessibles($partie->phase);
            if (!empty($casesAccessibles)) {
                $advPos = $adversaire->getPos();
                // Filtrer les cases occupées par l'adversaire
                $casesFiltrees = array_filter($casesAccessibles, function($case) use ($advPos) {
                    return !($case['x'] == $advPos['x'] && $case['y'] == $advPos['y']);
                });
                
                if (!empty($casesFiltrees)) {
                    $casesFiltrees = array_values($casesFiltrees);
                    $caseChoisie = $casesFiltrees[array_rand($casesFiltrees)];
                    return ['type' => 'bouger', 'x' => $caseChoisie['x'], 'y' => $caseChoisie['y']];
                }
            }
        } elseif ($choix <= 85 && $partie->phase === 2 && $vaisseau->bloqueTir == 0) {
            // Attaque en phase 2 (si énergie suffisante)
            if ($vaisseau->getEnergie() >= 2) {
                $typeArme = ($vaisseau->getEnergie() >= 3 && rand(1, 100) <= 40) ? 'plasma' : 'laser';
                return ['type' => 'attaque', 'arme' => $typeArme];
            }
        }
        
        // Par défaut : attendre
        return ['type' => 'attendre'];
    }
    
    /**
     * IA MOYEN : Stratégie équilibrée avec scoring
     */
    private function choisirActionMoyen($partie, $vaisseau, $adversaire, $idJoueur) {
        $actions = [];
        
        // 1. Évaluer les déplacements
        $casesAccessibles = $vaisseau->getCasesAccessibles($partie->phase);
        $advPos = $adversaire->getPos();
        
        foreach ($casesAccessibles as $case) {
            // Ignorer la case de l'adversaire
            if ($case['x'] == $advPos['x'] && $case['y'] == $advPos['y']) continue;
            
            $score = $this->evaluerDeplacement($case['x'], $case['y'], $partie, $vaisseau, $adversaire);
            $actions[] = [
                'type' => 'bouger',
                'x' => $case['x'],
                'y' => $case['y'],
                'score' => $score + rand(-15, 15) // Aléatoire pour variété
            ];
        }
        
        // 2. Évaluer l'attaque (phase 2 uniquement)
        if ($partie->phase === 2 && $vaisseau->bloqueTir == 0) {
            if ($vaisseau->getEnergie() >= 2) {
                $scoreLaser = 35 + rand(-10, 10);
                $actions[] = ['type' => 'attaque', 'arme' => 'laser', 'score' => $scoreLaser];
            }
            if ($vaisseau->getEnergie() >= 3) {
                $scorePlasma = 40 + rand(-10, 10);
                $actions[] = ['type' => 'attaque', 'arme' => 'plasma', 'score' => $scorePlasma];
            }
        }
        
        // 3. Évaluer les drones
        if (!empty($vaisseau->dronesStock)) {
            foreach ($vaisseau->dronesStock as $idx => $drone) {
                $scoreDrone = $this->evaluerDrone($drone, $vaisseau);
                $actions[] = ['type' => 'drone', 'idx' => $idx, 'score' => $scoreDrone];
            }
        }
        
        // 4. Évaluer les compétences
        $actionsComp = $this->evaluerCompetences($partie, $vaisseau, $adversaire);
        $actions = array_merge($actions, $actionsComp);
        
        // 5. Évaluer la recharge
        if ($vaisseau->getEnergie() < 5) {
            $actions[] = ['type' => 'attendre', 'score' => 30];
        } else {
            $actions[] = ['type' => 'attendre', 'score' => 5];
        }
        
        // Choisir l'action avec le meilleur score
        if (empty($actions)) {
            return ['type' => 'attendre'];
        }
        
        usort($actions, function($a, $b) {
            return $b['score'] - $a['score'];
        });
        
        return $actions[0];
    }
    
    /**
     * IA DIFFICILE : Optimisation avancée et anticipation
     */
    private function choisirActionDifficile($partie, $vaisseau, $adversaire, $idJoueur) {
        $actions = [];
        $posVaisseau = $vaisseau->getPos();
        $posAdv = $adversaire->getPos();
        
        // Analyser l'état de la partie
        $pvVaisseau = $vaisseau->getPv();
        $pvAdversaire = $adversaire->getPv();
        $energieVaisseau = $vaisseau->getEnergie();
        
        // Déterminer le mode de jeu
        $modeAgressif = ($pvAdversaire < 40 && $energieVaisseau >= 3);
        $modeDefensif = ($pvVaisseau < 30);
        
        // 1. PRIORITÉ MAXIMALE : Attaque létale si possible
        if ($partie->phase === 2 && $vaisseau->bloqueTir == 0 && $pvAdversaire <= 35 && $energieVaisseau >= 3) {
            $actions[] = ['type' => 'attaque', 'arme' => 'plasma', 'score' => 150];
        }
        
        // 2. Évaluer les déplacements avec stratégie avancée
        $casesAccessibles = $vaisseau->getCasesAccessibles($partie->phase);
        
        foreach ($casesAccessibles as $case) {
            if ($case['x'] == $posAdv['x'] && $case['y'] == $posAdv['y']) continue;
            
            $scoreBase = $this->evaluerDeplacement($case['x'], $case['y'], $partie, $vaisseau, $adversaire);
            
            // Bonus stratégiques
            $distance = abs($case['x'] - $posAdv['x']) + abs($case['y'] - $posAdv['y']);
            
            if ($modeAgressif) {
                // En mode agressif, se rapprocher mais garder distance sûre
                if ($distance >= 2 && $distance <= 4) $scoreBase += 30;
            } elseif ($modeDefensif) {
                // En mode défensif, garder distance
                if ($distance >= 5) $scoreBase += 40;
            } else {
                // Mode équilibré : distance optimale 3-4
                if ($distance >= 3 && $distance <= 4) $scoreBase += 25;
            }
            
            $actions[] = [
                'type' => 'bouger',
                'x' => $case['x'],
                'y' => $case['y'],
                'score' => $scoreBase
            ];
        }
        
        // 3. Évaluer les attaques
        if ($partie->phase === 2 && $vaisseau->bloqueTir == 0) {
            if ($energieVaisseau >= 3) {
                $scorePlasma = 60 + ($modeAgressif ? 20 : 0);
                $actions[] = ['type' => 'attaque', 'arme' => 'plasma', 'score' => $scorePlasma];
            }
            if ($energieVaisseau >= 2) {
                $scoreLaser = 50 + ($modeAgressif ? 15 : 0);
                $actions[] = ['type' => 'attaque', 'arme' => 'laser', 'score' => $scoreLaser];
            }
        }
        
        // 4. Évaluer les drones (stratégique)
        if (!empty($vaisseau->dronesStock)) {
            foreach ($vaisseau->dronesStock as $idx => $drone) {
                $scoreDrone = $this->evaluerDrone($drone, $vaisseau);
                
                // Bonus selon le contexte
                if ($drone->type === 'def' && $modeDefensif) $scoreDrone += 30;
                if ($drone->type === 'atk' && $modeAgressif) $scoreDrone += 25;
                if ($drone->type === 'rep' && $pvVaisseau < 50) $scoreDrone += 40;
                
                $actions[] = ['type' => 'drone', 'idx' => $idx, 'score' => $scoreDrone];
            }
        }
        
        // 5. Évaluer les compétences (priorité haute)
        $actionsComp = $this->evaluerCompetences($partie, $vaisseau, $adversaire);
        foreach ($actionsComp as &$comp) {
            $comp['score'] += 20; // Bonus pour l'IA difficile
        }
        $actions = array_merge($actions, $actionsComp);
        
        // 6. Recharge stratégique
        if ($energieVaisseau < 4) {
            $actions[] = ['type' => 'attendre', 'score' => 45];
        } elseif ($energieVaisseau < 7 && !$modeAgressif) {
            $actions[] = ['type' => 'attendre', 'score' => 20];
        } else {
            $actions[] = ['type' => 'attendre', 'score' => 5];
        }
        
        // Choisir la meilleure action
        if (empty($actions)) {
            return ['type' => 'attendre'];
        }
        
        usort($actions, function($a, $b) {
            return $b['score'] - $a['score'];
        });
        
        return $actions[0];
    }
    
    /**
     * Évaluer un déplacement
     */
    private function evaluerDeplacement($x, $y, $partie, $vaisseau, $adversaire) {
        $score = 0;
        
        // Distance à l'adversaire
        $advPos = $adversaire->getPos();
        $distance = abs($x - $advPos['x']) + abs($y - $advPos['y']);
        
        // Distance optimale : 3-4 cases
        if ($distance >= 3 && $distance <= 4) {
            $score += 20;
        } elseif ($distance < 2) {
            $score -= 15; // Trop proche, dangereux
        }
        
        // Vérifier les bonus à cette position
        foreach ($partie->bonusListe as $bonus) {
            if ($bonus->x == $x && $bonus->y == $y) {
                if ($bonus->type === 'energie') $score += 40;
                elseif ($bonus->type === 'personnage') $score += 45;
                elseif ($bonus->type === 'drone') $score += 35;
            }
        }
        
        // Bonus selon la position stratégique (centre du plateau)
        $distanceCentre = abs($x - 3.5) + abs($y - 3.5);
        $score += (7 - $distanceCentre) * 2;
        
        return $score;
    }
    
    /**
     * Évaluer un drone
     */
    private function evaluerDrone($drone, $vaisseau) {
        $score = 30;
        
        if ($drone->type === 'def' && $vaisseau->bouclier == 0) {
            $score += 25;
        }
        if ($drone->type === 'atk') {
            $score += 20;
        }
        if ($drone->type === 'rep' && $vaisseau->getPv() < 70) {
            $score += 30 + (70 - $vaisseau->getPv()) / 2;
        }
        if ($drone->type === 'esq') {
            $score += 15;
        }
        
        return $score;
    }
    
    /**
     * Évaluer les compétences disponibles
     */
    private function evaluerCompetences($partie, $vaisseau, $adversaire) {
        $actions = [];
        
        if ($partie->phase !== 2) return $actions;
        
        // Mentaliste : Entrave totale
        $idxMentaliste = $vaisseau->aUnMembrePret('Mentaliste');
        if ($idxMentaliste > -1) {
            $actions[] = ['type' => 'competence', 'comp' => 'mentaliste', 'score' => 70];
        }
        
        // Opérateur : Brouillage
        $idxOperateur = $vaisseau->aUnMembrePret('Operateur');
        if ($idxOperateur > -1) {
            $actions[] = ['type' => 'competence', 'comp' => 'operateur', 'score' => 65];
        }
        
        // Chasseur : Assassinat
        $idxChasseur = $vaisseau->aUnMembrePret('Chasseur');
        if ($idxChasseur > -1 && !empty($adversaire->getEquipage())) {
            $actions[] = ['type' => 'competence', 'comp' => 'chasseur', 'cible' => 0, 'score' => 75];
        }
        
        return $actions;
    }
    
    /**
     * Exécuter l'action choisie
     */
    private function executerAction($partie, $vaisseau, $adversaire, $action, $idJoueur) {
        switch ($action['type']) {
            case 'bouger':
                $advPos = $adversaire->getPos();
                $res = $vaisseau->seDeplacer($action['x'], $action['y'], $partie->phase, $advPos['x'], $advPos['y']);
                if ($res === true) {
                    $partie->verifierBonus($vaisseau);
                    $partie->nouveauMessage("L'IA se déplace.", 0);
                    $partie->passerTour();
                } else {
                    // Fallback si le déplacement échoue
                    $vaisseau->rechargerEnergie(3);
                    $partie->nouveauMessage("L'IA se repose (+3⚡).", 0);
                    $partie->passerTour();
                }
                break;
                
            case 'attaque':
                $pv = $vaisseau->tirer($adversaire, $action['arme']);
                if ($pv > -1) {
                    $cible = ($idJoueur == 1) ? 2 : 1;
                    $arme = ($action['arme'] == 'laser') ? "Laser" : "Plasma";
                    $partie->nouveauMessage("💥 TIR RÉUSSI ! $arme. Joueur $cible a perdu $pv PV.", 0);
                    $partie->verifierFinDePartie();
                    if (!$partie->gagnant) $partie->passerTour();
                } else {
                    $partie->nouveauMessage("💨 TIR MANQUÉ par l'IA.", 0);
                    $partie->passerTour();
                }
                break;
                
            case 'drone':
                if ($vaisseau->deployerDrone($action['idx'])) {
                    $partie->nouveauMessage("🛸 L'IA a déployé un drone !", 0);
                    $partie->passerTour();
                } else {
                    // Fallback
                    $vaisseau->rechargerEnergie(3);
                    $partie->nouveauMessage("L'IA se repose (+3⚡).", 0);
                    $partie->passerTour();
                }
                break;
                
            case 'competence':
                if ($action['comp'] === 'mentaliste') {
                    $idx = $vaisseau->aUnMembrePret('Mentaliste');
                    if ($idx > -1) {
                        $vaisseau->utiliserMentaliste($adversaire, $idx);
                        $partie->nouveauMessage("🔮 ENTRAVE TOTALE par l'IA (2 tours) !", 0);
                        $partie->passerTour();
                    }
                } elseif ($action['comp'] === 'operateur') {
                    $idx = $vaisseau->aUnMembrePret('Operateur');
                    if ($idx > -1) {
                        $vaisseau->utiliserOperateur($adversaire, $idx);
                        $partie->nouveauMessage("📡 BROUILLAGE ARMES par l'IA (2 tours) !", 0);
                        $partie->passerTour();
                    }
                } elseif ($action['comp'] === 'chasseur') {
                    $idxC = $vaisseau->aUnMembrePret('Chasseur');
                    if ($idxC > -1 && $adversaire->tuerMembre($action['cible'])) {
                        $vaisseau->utiliserCompetence($idxC);
                        $partie->nouveauMessage("🔫 ASSASSINAT RÉUSSI par l'IA !", 0);
                        $partie->passerTour();
                    }
                }
                break;
                
            case 'attendre':
            default:
                $vaisseau->rechargerEnergie(3);
                $partie->nouveauMessage("L'IA se repose (+3⚡).", 0);
                $partie->passerTour();
                break;
        }
    }
}
?>
