<?php
declare(strict_types=1);

// Point d'entrée principal pour cPanel
require_once __DIR__ . '/../config/cpanel_bootstrap.php';

use App\Core\Router;
use App\Core\Request;
use App\Core\Response;

try {
    // Test de connexion à la base de données
    if (!\App\Core\CpanelDatabase::testConnection()) {
        throw new \Exception('Impossible de se connecter à la base de données');
    }
    
    $router = new Router();
    
    // Routes publiques
    $router->get('/', 'HomeController@index');
    $router->get('/login', 'AuthController@showLogin');
    $router->post('/login', 'AuthController@login');
    $router->get('/register', 'AuthController@showRegister');
    $router->post('/register', 'AuthController@register');
    $router->get('/logout', 'AuthController@logout');
    
    // Routes spectacles
    $router->get('/performances', 'PerformanceController@index');
    $router->get('/performance/{id}', 'PerformanceController@show');
    $router->get('/reservation/{id}', 'ReservationController@show');
    $router->post('/reservation', 'ReservationController@store');
    
    // Routes utilisateur
    $router->get('/dashboard', 'UserController@dashboard');
    $router->get('/reviews', 'ReviewController@index');
    $router->post('/reviews', 'ReviewController@store');
    
    // Routes admin (protégées)
    $router->group('/admin', function($router) {
        $router->get('/', 'Admin\DashboardController@index');
        $router->get('/performances', 'Admin\PerformanceController@index');
        $router->get('/performances/create', 'Admin\PerformanceController@create');
        $router->post('/performances', 'Admin\PerformanceController@store');
        $router->get('/performances/{id}', 'Admin\PerformanceController@show');
        $router->get('/performances/{id}/edit', 'Admin\PerformanceController@edit');
        $router->post('/performances/{id}/update', 'Admin\PerformanceController@update');
        $router->post('/performances/{id}/delete', 'Admin\PerformanceController@destroy');
        
        $router->get('/users', 'Admin\UserController@index');
        $router->get('/reservations', 'Admin\ReservationController@index');
        $router->get('/reviews', 'Admin\ReviewController@index');
        $router->get('/statistics', 'Admin\StatisticsController@index');
    }, ['middleware' => 'admin']);
    
    $request = Request::createFromGlobals();
    $response = $router->dispatch($request);
    $response->send();
    
} catch (Exception $e) {
    error_log('Erreur application: ' . $e->getMessage());
    
    if (defined('APP_DEBUG') && APP_DEBUG) {
        echo '<h1>Erreur de débogage</h1>';
        echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
        echo '<pre>' . htmlspecialchars($e->getTraceAsString()) . '</pre>';
    } else {
        http_response_code(500);
        echo '<!DOCTYPE html>
        <html>
        <head><title>Erreur</title></head>
        <body>
            <h1>Une erreur est survenue</h1>
            <p>Nous nous excusons pour la gêne occasionnée. Veuillez réessayer plus tard.</p>
        </body>
        </html>';
    }
}
?>
