-- Création de la base de données pour Nouvel Acte
-- Base de données : ilanmolia_SAE202
-- Utilisateur : ilanmolia_molia

-- Table des rôles
CREATE TABLE roles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name ENUM('super_admin', 'manager', 'moderator', 'client') NOT NULL UNIQUE,
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Table des utilisateurs
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    display_name VARCHAR(100) NOT NULL,
    first_name VARCHAR(50),
    last_name VARCHAR(50),
    phone VARCHAR(20),
    avatar_url VARCHAR(500),
    role_id INT NOT NULL,
    email_verified_at TIMESTAMP NULL,
    is_active BOOLEAN DEFAULT TRUE,
    last_login_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (role_id) REFERENCES roles(id),
    INDEX idx_email (email),
    INDEX idx_role (role_id),
    INDEX idx_active (is_active)
);

-- Table des catégories
CREATE TABLE categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) NOT NULL UNIQUE,
    description TEXT,
    color VARCHAR(7) DEFAULT '#E30B17',
    icon VARCHAR(50),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_slug (slug),
    INDEX idx_active (is_active)
);

-- Table des spectacles
CREATE TABLE performances (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    slug VARCHAR(255) NOT NULL UNIQUE,
    description TEXT,
    short_description VARCHAR(500),
    duration INT NOT NULL COMMENT 'Durée en minutes',
    min_age TINYINT DEFAULT 0,
    max_capacity INT DEFAULT 100,
    category_id INT,
    poster_url VARCHAR(500),
    gallery JSON COMMENT 'URLs des images de la galerie',
    trailer_url VARCHAR(500),
    director VARCHAR(100),
    cast TEXT COMMENT 'Distribution',
    price_from DECIMAL(8,2) DEFAULT 0.00,
    price_to DECIMAL(8,2) DEFAULT 0.00,
    status ENUM('draft', 'online', 'archived', 'sold_out') DEFAULT 'draft',
    featured BOOLEAN DEFAULT FALSE,
    view_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id),
    INDEX idx_slug (slug),
    INDEX idx_status (status),
    INDEX idx_featured (featured),
    INDEX idx_category (category_id),
    FULLTEXT INDEX idx_search (title, description, short_description)
);

-- Table des salles
CREATE TABLE halls (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    seat_map JSON COMMENT 'Configuration des sièges en JSON',
    capacity INT NOT NULL,
    address TEXT,
    facilities JSON COMMENT 'Équipements disponibles',
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_active (is_active)
);

-- Table des séances/représentations
CREATE TABLE schedules (
    id INT PRIMARY KEY AUTO_INCREMENT,
    performance_id INT NOT NULL,
    hall_id INT NOT NULL,
    show_datetime DATETIME NOT NULL,
    end_datetime DATETIME,
    base_price DECIMAL(8,2) NOT NULL,
    available_seats INT,
    sold_seats INT DEFAULT 0,
    status ENUM('open', 'full', 'cancelled', 'finished') DEFAULT 'open',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (performance_id) REFERENCES performances(id) ON DELETE CASCADE,
    FOREIGN KEY (hall_id) REFERENCES halls(id),
    INDEX idx_show_datetime (show_datetime),
    INDEX idx_status (status),
    INDEX idx_performance (performance_id)
);

-- Table des sièges (si gestion siège par siège)
CREATE TABLE seats (
    id INT PRIMARY KEY AUTO_INCREMENT,
    hall_id INT NOT NULL,
    row_name CHAR(2) NOT NULL,
    seat_number SMALLINT NOT NULL,
    seat_type ENUM('standard', 'premium', 'vip', 'disabled') DEFAULT 'standard',
    price_modifier DECIMAL(5,2) DEFAULT 1.00 COMMENT 'Multiplicateur de prix',
    is_active BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (hall_id) REFERENCES halls(id),
    UNIQUE KEY unique_seat (hall_id, row_name, seat_number),
    INDEX idx_hall_type (hall_id, seat_type)
);

-- Table des réservations
CREATE TABLE reservations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    reservation_number VARCHAR(20) UNIQUE,
    total_price DECIMAL(8,2) NOT NULL,
    discount_amount DECIMAL(8,2) DEFAULT 0.00,
    final_price DECIMAL(8,2) NOT NULL,
    booked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    status ENUM('pending', 'confirmed', 'paid', 'cancelled', 'refunded') DEFAULT 'pending',
    payment_method VARCHAR(50),
    payment_ref VARCHAR(255),
    payment_date TIMESTAMP NULL,
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES users(id),
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_reservation_number (reservation_number),
    INDEX idx_booked_at (booked_at)
);

-- Table des éléments de réservation
CREATE TABLE reservation_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    reservation_id INT NOT NULL,
    schedule_id INT NOT NULL,
    seat_id INT NULL COMMENT 'NULL si pas de siège spécifique',
    quantity TINYINT DEFAULT 1,
    unit_price DECIMAL(8,2) NOT NULL,
    total_price DECIMAL(8,2) NOT NULL,
    status ENUM('active', 'cancelled') DEFAULT 'active',
    FOREIGN KEY (reservation_id) REFERENCES reservations(id) ON DELETE CASCADE,
    FOREIGN KEY (schedule_id) REFERENCES schedules(id),
    FOREIGN KEY (seat_id) REFERENCES seats(id),
    INDEX idx_reservation (reservation_id),
    INDEX idx_schedule (schedule_id)
);

-- Table des codes promo
CREATE TABLE coupons (
    id INT PRIMARY KEY AUTO_INCREMENT,
    code VARCHAR(50) NOT NULL UNIQUE,
    name VARCHAR(100),
    description VARCHAR(255),
    type ENUM('percentage', 'fixed_amount') DEFAULT 'percentage',
    value DECIMAL(8,2) NOT NULL COMMENT 'Pourcentage ou montant fixe',
    min_amount DECIMAL(8,2) DEFAULT 0.00 COMMENT 'Montant minimum pour utiliser le coupon',
    max_discount DECIMAL(8,2) NULL COMMENT 'Réduction maximum',
    max_redemptions INT DEFAULT 1,
    current_redemptions INT DEFAULT 0,
    valid_from TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_code (code),
    INDEX idx_active (is_active),
    INDEX idx_expires (expires_at)
);

-- Table de liaison réservations-coupons
CREATE TABLE reservation_coupons (
    reservation_id INT,
    coupon_id INT,
    discount_amount DECIMAL(8,2) NOT NULL,
    applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (reservation_id, coupon_id),
    FOREIGN KEY (reservation_id) REFERENCES reservations(id) ON DELETE CASCADE,
    FOREIGN KEY (coupon_id) REFERENCES coupons(id)
);

-- Table des avis
CREATE TABLE reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    performance_id INT NOT NULL,
    rating TINYINT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    title VARCHAR(200),
    comment TEXT,
    is_approved BOOLEAN DEFAULT FALSE,
    admin_response TEXT NULL,
    helpful_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (performance_id) REFERENCES performances(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_performance (user_id, performance_id),
    INDEX idx_performance_approved (performance_id, is_approved),
    INDEX idx_rating (rating)
);

-- Table des favoris
CREATE TABLE favorites (
    user_id INT,
    performance_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (user_id, performance_id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (performance_id) REFERENCES performances(id) ON DELETE CASCADE
);

-- Table des amis
CREATE TABLE friends (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    friend_id INT NOT NULL,
    status ENUM('requested', 'accepted', 'blocked') DEFAULT 'requested',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (friend_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_friendship (user_id, friend_id),
    CHECK (user_id != friend_id),
    INDEX idx_user_status (user_id, status)
);

-- Table des groupes
CREATE TABLE groups (
    id INT PRIMARY KEY AUTO_INCREMENT,
    owner_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    avatar_url VARCHAR(500),
    is_private BOOLEAN DEFAULT FALSE,
    member_count INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id),
    INDEX idx_owner (owner_id),
    INDEX idx_private (is_private)
);

-- Table des membres de groupes
CREATE TABLE group_members (
    group_id INT,
    user_id INT,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    role ENUM('member', 'admin') DEFAULT 'member',
    PRIMARY KEY (group_id, user_id),
    FOREIGN KEY (group_id) REFERENCES groups(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Table des logs d'audit
CREATE TABLE logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NULL,
    action ENUM('create', 'update', 'delete', 'login', 'logout', 'view', 'purchase') NOT NULL,
    entity VARCHAR(50) NOT NULL COMMENT 'Table concernée',
    entity_id INT NULL COMMENT 'ID de l\'entité concernée',
    old_values JSON NULL,
    new_values JSON NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_action (user_id, action),
    INDEX idx_entity (entity, entity_id),
    INDEX idx_created_at (created_at)
);

-- ========================================
-- INSERTION DES DONNÉES DE BASE
-- ========================================

-- Insertion des rôles
INSERT INTO roles (name, description) VALUES 
('super_admin', 'Administrateur principal avec tous les droits'),
('manager', 'Gestionnaire avec droits étendus'),
('moderator', 'Modérateur pour la gestion du contenu'),
('client', 'Client standard');

-- Insertion des catégories
INSERT INTO categories (name, slug, description, color, icon) VALUES
('Théâtre', 'theatre', 'Pièces de théâtre classiques et contemporaines', '#8B4513', 'theater'),
('Comédie musicale', 'comedie-musicale', 'Spectacles musicaux et comédies musicales', '#FF6B6B', 'music'),
('Danse', 'danse', 'Spectacles de danse classique et moderne', '#4ECDC4', 'dance'),
('Concert', 'concert', 'Concerts et récitals musicaux', '#45B7D1', 'microphone'),
('Humour', 'humour', 'Spectacles d\'humour et one-man-show', '#96CEB4', 'smile'),
('Opéra', 'opera', 'Opéras et spectacles lyriques', '#FFEAA7', 'crown'),
('Cirque', 'cirque', 'Spectacles de cirque et arts du spectacle', '#DDA0DD', 'star');

-- Création d'un utilisateur admin par défaut
INSERT INTO users (email, password_hash, display_name, first_name, last_name, role_id, email_verified_at, is_active) VALUES
('admin@nouvelacte.fr', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrateur', 'Admin', 'Système', 1, NOW(), TRUE),
('manager@nouvelacte.fr', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Gestionnaire', 'Manager', 'Théâtre', 2, NOW(), TRUE),
('client@nouvelacte.fr', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Client Test', 'Jean', 'Dupont', 4, NOW(), TRUE);

-- Création des salles
INSERT INTO halls (name, description, capacity, seat_map, address, facilities) VALUES
('Salle Principale', 'Grande salle de spectacle avec excellente acoustique', 300, 
'{"rows": [
    {"name": "A", "seats": 20, "type": "premium"},
    {"name": "B", "seats": 20, "type": "premium"},
    {"name": "C", "seats": 25, "type": "standard"},
    {"name": "D", "seats": 25, "type": "standard"},
    {"name": "E", "seats": 25, "type": "standard"},
    {"name": "F", "seats": 30, "type": "standard"}
]}', 
'123 Avenue du Théâtre, Paris', 
'["climatisation", "accessibilité PMR", "bar", "vestiaire"]'),

('Studio Intime', 'Petite salle pour spectacles intimistes', 80,
'{"rows": [
    {"name": "A", "seats": 10, "type": "premium"},
    {"name": "B", "seats": 12, "type": "standard"},
    {"name": "C", "seats": 12, "type": "standard"},
    {"name": "D", "seats": 15, "type": "standard"},
    {"name": "E", "seats": 15, "type": "standard"}
]}',
'456 Rue de l\'Art, Paris',
'["climatisation", "accessibilité PMR", "éclairage modulable"]'),

('Amphithéâtre', 'Grande salle en amphithéâtre pour concerts', 500,
'{"rows": [
    {"name": "Orchestre", "seats": 100, "type": "premium"},
    {"name": "Mezzanine", "seats": 150, "type": "standard"},
    {"name": "Balcon", "seats": 250, "type": "standard"}
]}',
'789 Boulevard des Arts, Paris',
'["sonorisation professionnelle", "éclairage scénique", "loges artistes", "bar", "restaurant"]');

-- Insertion des sièges pour la salle principale
INSERT INTO seats (hall_id, row_name, seat_number, seat_type, price_modifier) VALUES
-- Rangée A (Premium)
(1, 'A', 1, 'premium', 1.5), (1, 'A', 2, 'premium', 1.5), (1, 'A', 3, 'premium', 1.5), (1, 'A', 4, 'premium', 1.5), (1, 'A', 5, 'premium', 1.5),
(1, 'A', 6, 'premium', 1.5), (1, 'A', 7, 'premium', 1.5), (1, 'A', 8, 'premium', 1.5), (1, 'A', 9, 'premium', 1.5), (1, 'A', 10, 'premium', 1.5),
(1, 'A', 11, 'premium', 1.5), (1, 'A', 12, 'premium', 1.5), (1, 'A', 13, 'premium', 1.5), (1, 'A', 14, 'premium', 1.5), (1, 'A', 15, 'premium', 1.5),
(1, 'A', 16, 'premium', 1.5), (1, 'A', 17, 'premium', 1.5), (1, 'A', 18, 'premium', 1.5), (1, 'A', 19, 'premium', 1.5), (1, 'A', 20, 'premium', 1.5),

-- Rangée B (Premium)
(1, 'B', 1, 'premium', 1.5), (1, 'B', 2, 'premium', 1.5), (1, 'B', 3, 'premium', 1.5), (1, 'B', 4, 'premium', 1.5), (1, 'B', 5, 'premium', 1.5),
(1, 'B', 6, 'premium', 1.5), (1, 'B', 7, 'premium', 1.5), (1, 'B', 8, 'premium', 1.5), (1, 'B', 9, 'premium', 1.5), (1, 'B', 10, 'premium', 1.5),
(1, 'B', 11, 'premium', 1.5), (1, 'B', 12, 'premium', 1.5), (1, 'B', 13, 'premium', 1.5), (1, 'B', 14, 'premium', 1.5), (1, 'B', 15, 'premium', 1.5),
(1, 'B', 16, 'premium', 1.5), (1, 'B', 17, 'premium', 1.5), (1, 'B', 18, 'premium', 1.5), (1, 'B', 19, 'premium', 1.5), (1, 'B', 20, 'premium', 1.5);

-- Insertion des spectacles d'exemple
INSERT INTO performances (title, slug, description, short_description, duration, min_age, category_id, director, cast, price_from, price_to, status, featured) VALUES

('Roméo et Juliette', 'romeo-et-juliette', 
'La plus célèbre histoire d\'amour de Shakespeare prend vie sur notre scène. Une mise en scène moderne qui revisite ce classique intemporel avec une distribution exceptionnelle. Entre passion et tragédie, laissez-vous emporter par cette œuvre magistrale qui continue de toucher les cœurs après plus de 400 ans.',
'La plus célèbre histoire d\'amour de Shakespeare dans une mise en scène moderne et captivante.',
150, 12, 1, 'Marie Dubois', 'Pierre Martin, Sophie Laurent, Jean-Luc Moreau', 25.00, 65.00, 'online', TRUE),

('Les Misérables - Comédie Musicale', 'les-miserables-comedie-musicale',
'L\'adaptation musicale du chef-d\'œuvre de Victor Hugo. Suivez Jean Valjean dans sa quête de rédemption à travers la France du XIXe siècle. Avec des chansons inoubliables et une mise en scène grandiose, ce spectacle vous transportera dans l\'univers poignant des Misérables.',
'L\'adaptation musicale légendaire du chef-d\'œuvre de Victor Hugo.',
180, 10, 2, 'Claude Lelouch', 'Ensemble de 25 artistes', 35.00, 85.00, 'online', TRUE),

('Casse-Noisette', 'casse-noisette',
'Le ballet féerique de Tchaïkovski dans une version revisitée. Plongez dans l\'univers magique de Clara et de son Casse-Noisette. Une chorégraphie moderne sur la musique intemporelle du compositeur russe, interprétée par les meilleurs danseurs de la compagnie.',
'Le ballet féerique de Tchaïkovski dans une version moderne et enchanteresse.',
120, 6, 3, 'Anna Petrov', 'Corps de ballet de l\'Opéra', 30.00, 70.00, 'online', TRUE),

('Concert Jazz Quartet', 'concert-jazz-quartet',
'Une soirée jazz exceptionnelle avec le quartet le plus en vue de la scène parisienne. Standards du jazz, compositions originales et improvisations magistrales vous attendent pour une soirée musicale inoubliable dans l\'intimité de notre studio.',
'Soirée jazz exceptionnelle avec le quartet le plus en vue de Paris.',
90, 16, 4, 'Marcus Johnson', 'Jazz Quartet Paris', 20.00, 45.00, 'online', FALSE),

('Gad Elmaleh - Nouveau Spectacle', 'gad-elmaleh-nouveau-spectacle',
'Le maître de l\'humour français revient avec un nouveau spectacle hilarant. Entre observations du quotidien et anecdotes personnelles, Gad Elmaleh vous garantit une soirée de rire et de bonne humeur. Un spectacle inédit qui promet de marquer les esprits.',
'Le maître de l\'humour français dans son nouveau spectacle hilarant.',
105, 14, 5, 'Gad Elmaleh', 'Gad Elmaleh', 28.00, 55.00, 'online', TRUE),

('La Traviata', 'la-traviata',
'L\'opéra de Verdi dans une production somptueuse. L\'histoire tragique de Violetta Valéry portée par des voix d\'exception et une mise en scène raffinée. Un spectacle lyrique de haute volée qui ravira les amateurs d\'opéra comme les néophytes.',
'L\'opéra de Verdi dans une production somptueuse avec des voix d\'exception.',
165, 12, 6, 'Roberto Alagna', 'Chœur et orchestre de l\'Opéra', 40.00, 120.00, 'online', FALSE),

('Cirque Plume - Spectacle Poétique', 'cirque-plume-spectacle-poetique',
'Un spectacle unique mêlant cirque, musique et poésie. Le Cirque Plume vous invite dans un univers onirique où les prouesses acrobatiques se mêlent aux mélodies envoûtantes. Une expérience artistique totale pour toute la famille.',
'Spectacle unique mêlant cirque, musique et poésie dans un univers onirique.',
75, 4, 7, 'Philippe Découflé', 'Troupe du Cirque Plume', 22.00, 48.00, 'online', TRUE);

-- Insertion des séances
INSERT INTO schedules (performance_id, hall_id, show_datetime, end_datetime, base_price, available_seats, status) VALUES
-- Roméo et Juliette
(1, 1, '2024-02-15 20:00:00', '2024-02-15 22:30:00', 45.00, 300, 'open'),
(1, 1, '2024-02-16 20:00:00', '2024-02-16 22:30:00', 45.00, 300, 'open'),
(1, 1, '2024-02-17 15:00:00', '2024-02-17 17:30:00', 40.00, 300, 'open'),

-- Les Misérables
(2, 3, '2024-02-20 19:30:00', '2024-02-20 22:30:00', 60.00, 500, 'open'),
(2, 3, '2024-02-21 19:30:00', '2024-02-21 22:30:00', 60.00, 500, 'open'),

-- Casse-Noisette
(3, 1, '2024-02-25 16:00:00', '2024-02-25 18:00:00', 50.00, 300, 'open'),
(3, 1, '2024-02-25 20:30:00', '2024-02-25 22:30:00', 55.00, 300, 'open'),

-- Concert Jazz
(4, 2, '2024-02-28 21:00:00', '2024-02-28 22:30:00', 32.00, 80, 'open'),

-- Gad Elmaleh
(5, 1, '2024-03-05 20:30:00', '2024-03-05 22:15:00', 42.00, 300, 'open'),
(5, 1, '2024-03-06 20:30:00', '2024-03-06 22:15:00', 42.00, 300, 'open'),

-- La Traviata
(6, 3, '2024-03-10 19:00:00', '2024-03-10 21:45:00', 80.00, 500, 'open'),

-- Cirque Plume
(7, 1, '2024-03-15 15:00:00', '2024-03-15 16:15:00', 35.00, 300, 'open'),
(7, 1, '2024-03-15 18:00:00', '2024-03-15 19:15:00', 35.00, 300, 'open');

-- Insertion de codes promo d'exemple
INSERT INTO coupons (code, name, description, type, value, min_amount, max_redemptions, expires_at) VALUES
('BIENVENUE10', 'Bienvenue', 'Réduction de 10% pour les nouveaux clients', 'percentage', 10.00, 20.00, 100, '2024-12-31 23:59:59'),
('ETUDIANT15', 'Tarif Étudiant', 'Réduction de 15% pour les étudiants', 'percentage', 15.00, 15.00, 200, '2024-12-31 23:59:59'),
('FAMILLE20', 'Tarif Famille', 'Réduction de 20€ pour les familles (4 places minimum)', 'fixed_amount', 20.00, 100.00, 50, '2024-12-31 23:59:59'),
('FIDELITE25', 'Client Fidèle', 'Réduction de 25% pour les clients fidèles', 'percentage', 25.00, 50.00, 30, '2024-12-31 23:59:59');

-- Insertion d'avis d'exemple
INSERT INTO reviews (user_id, performance_id, rating, title, comment, is_approved) VALUES
(3, 1, 5, 'Magnifique adaptation !', 'Une mise en scène moderne qui respecte l\'esprit de Shakespeare. Les acteurs sont exceptionnels et la scénographie est à couper le souffle. Je recommande vivement !', TRUE),
(3, 2, 4, 'Spectacle grandiose', 'Les Misérables dans une version musicale époustouflante. Quelques longueurs mais l\'émotion est au rendez-vous. Les voix sont magnifiques.', TRUE),
(3, 3, 5, 'Féerique pour toute la famille', 'Mes enfants ont adoré et moi aussi ! La chorégraphie est sublime et la musique de Tchaïkovski toujours aussi envoûtante.', TRUE);

-- Insertion d'une réservation d'exemple
INSERT INTO reservations (user_id, reservation_number, total_price, discount_amount, final_price, status, payment_method) VALUES
(3, 'RES20240101001', 90.00, 9.00, 81.00, 'paid', 'carte_bancaire');

INSERT INTO reservation_items (reservation_id, schedule_id, quantity, unit_price, total_price) VALUES
(1, 1, 2, 45.00, 90.00);

-- Mise à jour des compteurs
UPDATE schedules SET sold_seats = 2, available_seats = 298 WHERE id = 1;

-- Insertion de logs d'exemple
INSERT INTO logs (user_id, action, entity, entity_id, ip_address) VALUES
(1, 'login', 'users', 1, '127.0.0.1'),
(3, 'purchase', 'reservations', 1, '192.168.1.100'),
(3, 'view', 'performances', 1, '192.168.1.100');

-- ========================================
-- VUES UTILES POUR LES STATISTIQUES
-- ========================================

-- Vue pour les statistiques des spectacles
CREATE VIEW performance_stats AS
SELECT 
    p.id,
    p.title,
    p.status,
    COUNT(DISTINCT s.id) as total_schedules,
    COUNT(DISTINCT r.id) as total_reservations,
    COALESCE(SUM(ri.quantity), 0) as tickets_sold,
    COALESCE(SUM(ri.total_price), 0) as revenue,
    COALESCE(AVG(rev.rating), 0) as avg_rating,
    COUNT(DISTINCT rev.id) as review_count
FROM performances p
LEFT JOIN schedules s ON p.id = s.performance_id
LEFT JOIN reservation_items ri ON s.id = ri.schedule_id
LEFT JOIN reservations r ON ri.reservation_id = r.id AND r.status = 'paid'
LEFT JOIN reviews rev ON p.id = rev.performance_id AND rev.is_approved = TRUE
GROUP BY p.id, p.title, p.status;

-- Vue pour les statistiques des utilisateurs
CREATE VIEW user_stats AS
SELECT 
    u.id,
    u.display_name,
    u.email,
    r.name as role_name,
    COUNT(DISTINCT res.id) as total_reservations,
    COALESCE(SUM(res.final_price), 0) as total_spent,
    COUNT(DISTINCT rev.id) as reviews_written,
    u.created_at as member_since
FROM users u
LEFT JOIN roles r ON u.role_id = r.id
LEFT JOIN reservations res ON u.id = res.user_id AND res.status = 'paid'
LEFT JOIN reviews rev ON u.id = rev.user_id
GROUP BY u.id, u.display_name, u.email, r.name, u.created_at;

-- ========================================
-- PROCÉDURES STOCKÉES UTILES
-- ========================================

DELIMITER //

-- Procédure pour calculer les statistiques du jour
CREATE PROCEDURE GetDailyStats(IN target_date DATE)
BEGIN
    SELECT 
        COUNT(DISTINCT r.id) as reservations_today,
        COALESCE(SUM(r.final_price), 0) as revenue_today,
        COUNT(DISTINCT ri.schedule_id) as shows_with_sales,
        COALESCE(SUM(ri.quantity), 0) as tickets_sold_today
    FROM reservations r
    LEFT JOIN reservation_items ri ON r.id = ri.reservation_id
    WHERE DATE(r.booked_at) = target_date 
    AND r.status = 'paid';
END //

-- Procédure pour obtenir les spectacles populaires
CREATE PROCEDURE GetPopularPerformances(IN limit_count INT)
BEGIN
    SELECT 
        p.*,
        c.name as category_name,
        ps.tickets_sold,
        ps.revenue,
        ps.avg_rating
    FROM performances p
    LEFT JOIN categories c ON p.category_id = c.id
    LEFT JOIN performance_stats ps ON p.id = ps.id
    WHERE p.status = 'online'
    ORDER BY ps.tickets_sold DESC, ps.avg_rating DESC
    LIMIT limit_count;
END //

DELIMITER ;

-- ========================================
-- INDEX SUPPLÉMENTAIRES POUR LES PERFORMANCES
-- ========================================

-- Index pour les recherches fréquentes
CREATE INDEX idx_performances_featured_status ON performances(featured, status);
CREATE INDEX idx_schedules_datetime_status ON schedules(show_datetime, status);
CREATE INDEX idx_reservations_user_status ON reservations(user_id, status);
CREATE INDEX idx_reviews_performance_approved ON reviews(performance_id, is_approved);

-- ========================================
-- TRIGGERS POUR MAINTENIR LA COHÉRENCE
-- ========================================

DELIMITER //

-- Trigger pour mettre à jour le nombre de sièges vendus
CREATE TRIGGER update_sold_seats_after_reservation
AFTER INSERT ON reservation_items
FOR EACH ROW
BEGIN
    UPDATE schedules 
    SET sold_seats = sold_seats + NEW.quantity,
        available_seats = capacity - (sold_seats + NEW.quantity)
    WHERE id = NEW.schedule_id;
END //

-- Trigger pour générer un numéro de réservation automatique
CREATE TRIGGER generate_reservation_number
BEFORE INSERT ON reservations
FOR EACH ROW
BEGIN
    IF NEW.reservation_number IS NULL THEN
        SET NEW.reservation_number = CONCAT('RES', DATE_FORMAT(NOW(), '%Y%m%d'), LPAD(LAST_INSERT_ID(), 3, '0'));
    END IF;
END //

DELIMITER ;

-- Message de fin
SELECT 'Base de données Nouvel Acte créée avec succès !' as message;
SELECT 'Utilisateurs créés:' as info;
SELECT email, display_name, r.name as role FROM users u JOIN roles r ON u.role_id = r.id;
