<?php
declare(strict_types=1);

namespace App\Models;

use App\Core\Database;
use PDO;

class Performance
{
    private PDO $db;
    
    public function __construct()
    {
        $this->db = Database::getInstance();
    }
    
    public function getFeatured(int $limit = 5): array
    {
        $stmt = $this->db->prepare('
            SELECT p.*, c.name as category_name,
                   AVG(r.rating) as avg_rating,
                   COUNT(r.id) as review_count
            FROM performances p
            LEFT JOIN categories c ON p.category_id = c.id
            LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
            WHERE p.status = "online"
            GROUP BY p.id
            ORDER BY p.created_at DESC
            LIMIT ?
        ');
        $stmt->execute([$limit]);
        
        return $stmt->fetchAll();
    }
    
    public function getPopular(int $limit = 10): array
    {
        $stmt = $this->db->prepare('
            SELECT p.*, c.name as category_name,
                   AVG(r.rating) as avg_rating,
                   COUNT(r.id) as review_count,
                   COUNT(DISTINCT res.id) as reservation_count
            FROM performances p
            LEFT JOIN categories c ON p.category_id = c.id
            LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
            LEFT JOIN schedules s ON p.id = s.performance_id
            LEFT JOIN reservation_items ri ON s.id = ri.schedule_id
            LEFT JOIN reservations res ON ri.reservation_id = res.id
            WHERE p.status = "online"
            GROUP BY p.id
            ORDER BY reservation_count DESC, avg_rating DESC
            LIMIT ?
        ');
        $stmt->execute([$limit]);
        
        return $stmt->fetchAll();
    }
    
    public function findById(int $id): ?array
    {
        $stmt = $this->db->prepare('
            SELECT p.*, c.name as category_name,
                   AVG(r.rating) as avg_rating,
                   COUNT(r.id) as review_count
            FROM performances p
            LEFT JOIN categories c ON p.category_id = c.id
            LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
            WHERE p.id = ?
            GROUP BY p.id
        ');
        $stmt->execute([$id]);
        
        $performance = $stmt->fetch();
        return $performance ?: null;
    }
    
    public function getActiveCount(): int
    {
        $stmt = $this->db->prepare('SELECT COUNT(*) FROM performances WHERE status = "online"');
        $stmt->execute();
        
        return (int) $stmt->fetchColumn();
    }
}
?>
