<?php
declare(strict_types=1);

namespace App\Core;

class Router
{
    private array $routes = [];
    private array $middlewares = [];
    
    public function get(string $path, string $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }
    
    public function post(string $path, string $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }
    
    public function put(string $path, string $handler): void
    {
        $this->addRoute('PUT', $path, $handler);
    }
    
    public function delete(string $path, string $handler): void
    {
        $this->addRoute('DELETE', $path, $handler);
    }
    
    public function resource(string $path, string $controller): void
    {
        $this->get($path, $controller . '@index');
        $this->get($path . '/create', $controller . '@create');
        $this->post($path, $controller . '@store');
        $this->get($path . '/{id}', $controller . '@show');
        $this->get($path . '/{id}/edit', $controller . '@edit');
        $this->put($path . '/{id}', $controller . '@update');
        $this->delete($path . '/{id}', $controller . '@destroy');
    }
    
    public function group(string $prefix, callable $callback, array $options = []): void
    {
        $oldMiddlewares = $this->middlewares;
        if (isset($options['middleware'])) {
            $this->middlewares[] = $options['middleware'];
        }
        
        $callback($this);
        $this->middlewares = $oldMiddlewares;
    }
    
    private function addRoute(string $method, string $path, string $handler): void
    {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'handler' => $handler,
            'middlewares' => $this->middlewares
        ];
    }
    
    public function dispatch(Request $request): Response
    {
        $method = $request->getMethod();
        $path = $request->getPath();
        
        foreach ($this->routes as $route) {
            if ($route['method'] === $method && $this->matchPath($route['path'], $path, $params)) {
                // Exécution des middlewares
                foreach ($route['middlewares'] as $middleware) {
                    $this->executeMiddleware($middleware, $request);
                }
                
                return $this->executeHandler($route['handler'], $params, $request);
            }
        }
        
        return new Response('Page non trouvée', 404);
    }
    
    private function matchPath(string $routePath, string $requestPath, &$params = []): bool
    {
        $params = [];
        $routePattern = preg_replace('/\{([^}]+)\}/', '([^/]+)', $routePath);
        $routePattern = '#^' . $routePattern . '$#';
        
        if (preg_match($routePattern, $requestPath, $matches)) {
            array_shift($matches);
            $params = $matches;
            return true;
        }
        
        return false;
    }
    
    private function executeMiddleware(string $middleware, Request $request): void
    {
        switch ($middleware) {
            case 'admin':
                if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
                    header('Location: /login');
                    exit;
                }
                break;
        }
    }
    
    private function executeHandler(string $handler, array $params, Request $request): Response
    {
        [$controllerName, $method] = explode('@', $handler);
        $controllerClass = 'App\\Controllers\\' . $controllerName;
        
        if (!class_exists($controllerClass)) {
            return new Response('Contrôleur non trouvé', 404);
        }
        
        $controller = new $controllerClass();
        
        if (!method_exists($controller, $method)) {
            return new Response('Méthode non trouvée', 404);
        }
        
        return $controller->$method($request, ...$params);
    }
}
?>
