<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Core\View;
use App\Models\CpanelUser;

class CpanelAuthController
{
    public function showLogin(Request $request): Response
    {
        if (isset($_SESSION['user'])) {
            return Response::redirect('/dashboard');
        }
        
        $content = View::render('auth/cpanel_login');
        return new Response($content);
    }
    
    public function login(Request $request): Response
    {
        $email = filter_var($request->post('email'), FILTER_SANITIZE_EMAIL);
        $password = $request->post('password');
        
        if (!$email || !$password) {
            $content = View::render('auth/cpanel_login', [
                'error' => 'Email et mot de passe requis'
            ]);
            return new Response($content);
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $content = View::render('auth/cpanel_login', [
                'error' => 'Format d\'email invalide'
            ]);
            return new Response($content);
        }
        
        $userModel = new CpanelUser();
        $user = $userModel->findByEmail($email);
        
        if ($user && password_verify($password, $user['password_hash'])) {
            // Régénération de l'ID de session pour sécurité
            session_regenerate_id(true);
            
            $_SESSION['user'] = [
                'id' => $user['id'],
                'email' => $user['email'],
                'display_name' => $user['display_name'],
                'role' => $user['role_name']
            ];
            
            // Log de connexion
            error_log("Connexion réussie pour: " . $email);
            
            $redirectUrl = ($user['role_name'] === 'super_admin' || $user['role_name'] === 'manager') 
                ? '/admin' 
                : '/dashboard';
                
            return Response::redirect($redirectUrl);
        }
        
        // Log de tentative de connexion échouée
        error_log("Tentative de connexion échouée pour: " . $email);
        
        $content = View::render('auth/cpanel_login', [
            'error' => 'Identifiants incorrects'
        ]);
        return new Response($content);
    }
    
    public function showRegister(Request $request): Response
    {
        $content = View::render('auth/cpanel_register');
        return new Response($content);
    }
    
    public function register(Request $request): Response
    {
        $email = filter_var($request->post('email'), FILTER_SANITIZE_EMAIL);
        $password = $request->post('password');
        $confirmPassword = $request->post('confirm_password');
        $displayName = filter_var($request->post('display_name'), FILTER_SANITIZE_STRING);
        
        // Validation
        $errors = [];
        
        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Email valide requis';
        }
        
        if (!$password || strlen($password) < 8) {
            $errors[] = 'Mot de passe de 8 caractères minimum requis';
        }
        
        if ($password !== $confirmPassword) {
            $errors[] = 'Les mots de passe ne correspondent pas';
        }
        
        if (!$displayName || strlen($displayName) < 2) {
            $errors[] = 'Nom d\'affichage requis (2 caractères minimum)';
        }
        
        if (!empty($errors)) {
            $content = View::render('auth/cpanel_register', [
                'errors' => $errors,
                'old' => $request->all()
            ]);
            return new Response($content);
        }
        
        $userModel = new CpanelUser();
        
        // Vérification email unique
        if ($userModel->findByEmail($email)) {
            $content = View::render('auth/cpanel_register', [
                'errors' => ['Cet email est déjà utilisé'],
                'old' => $request->all()
            ]);
            return new Response($content);
        }
        
        // Création de l'utilisateur
        $userId = $userModel->create([
            'email' => $email,
            'password_hash' => password_hash($password, PASSWORD_ARGON2ID),
            'display_name' => $displayName,
            'role_id' => 4 // Client par défaut
        ]);
        
        if ($userId) {
            error_log("Nouvel utilisateur créé: " . $email);
            return Response::redirect('/login?registered=1');
        }
        
        $content = View::render('auth/cpanel_register', [
            'errors' => ['Erreur lors de l\'inscription. Veuillez réessayer.'],
            'old' => $request->all()
        ]);
        return new Response($content);
    }
    
    public function logout(Request $request): Response
    {
        if (isset($_SESSION['user'])) {
            error_log("Déconnexion de: " . $_SESSION['user']['email']);
        }
        
        // Destruction complète de la session
        $_SESSION = [];
        
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        session_destroy();
        
        return Response::redirect('/');
    }
}
?>
