// JavaScript pour la page de spectacle

document.addEventListener("DOMContentLoaded", () => {
  initPerformancePage()
  initBookingButtons()
  // initFavoriteToggle(); // Déjà géré par l'attribut onclick et la fonction globale toggleFavorite
  initWatchlistToggle()
  initTrailerModal()
})

function initPerformancePage() {
  console.log("Page spectacle initialisée")
  // Charger les avis si nécessaire
  // loadReviews(); // Si les avis ne sont pas chargés par PHP
}

function initBookingButtons() {
  const bookingButtons = document.querySelectorAll('a[href*="reservation.php"]')
  bookingButtons.forEach((button) => {
    button.addEventListener("click", function (e) {
      const scheduleId = new URL(this.href).searchParams.get("schedule_id")
      if (scheduleId) {
        // checkAvailability(scheduleId, this.href); // Décommenter si checkAvailability est implémenté
        // e.preventDefault();
      }
    })
  })
}

// La fonction toggleFavorite est maintenant passée directement en onclick
// function initFavoriteToggle() { /* ... */ }

function toggleFavorite(buttonElement, performanceId) {
  fetch("api/favorites.php", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ performance_id: performanceId, action: "toggle" }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        buttonElement.classList.toggle("active", data.is_favorite)
        showGlobalNotification(data.is_favorite ? "Ajouté aux favoris !" : "Retiré des favoris.", "success")
      } else {
        showGlobalNotification(data.error || "Erreur lors de la mise à jour des favoris.", "error")
      }
    })
    .catch((error) => {
      console.error("Erreur toggleFavorite:", error)
      showGlobalNotification("Erreur de connexion pour les favoris.", "error")
    })
}

function initWatchlistToggle() {
  const watchlistButton = document.getElementById("btnToggleWatchlist")
  if (watchlistButton) {
    watchlistButton.addEventListener("click", function () {
      const performanceId = this.dataset.performanceId
      const eventDate = this.dataset.eventDate // Format YYYY-MM-DD HH:MM:SS

      if (!performanceId || !eventDate) {
        showGlobalNotification("Information de spectacle ou date manquante pour l'ajout à la liste.", "error")
        return
      }

      fetch("api/watchlist.php", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          performance_id: Number.parseInt(performanceId),
          event_date: eventDate,
        }),
      })
        .then((response) => response.json())
        .then((data) => {
          if (data.success) {
            const isActive = data.action === "added"
            this.classList.toggle("active", isActive)
            this.querySelector(".btn-watchlist-text").textContent = isActive ? 'Dans "À venir"' : "À venir"
            this.title = isActive ? 'Retirer de "À venir"' : 'Ajouter à "À venir"'
            showGlobalNotification(
              data.message || (isActive ? "Ajouté à la liste 'À venir'." : "Retiré de la liste 'À venir'."),
              "success",
            )
          } else {
            showGlobalNotification(data.error || "Erreur lors de la mise à jour de la liste 'À venir'.", "error")
          }
        })
        .catch((error) => {
          console.error("Erreur toggleWatchlist:", error)
          showGlobalNotification("Erreur de connexion pour la liste 'À venir'.", "error")
        })
    })
  }
}

function initTrailerModal() {
  const modal = document.getElementById("trailerModal")
  const iframe = document.getElementById("trailerIframe")
  const closeBtn = modal.querySelector(".close-trailer-modal")

  closeBtn.onclick = () => {
    modal.style.display = "none"
    iframe.src = "" // Stop video
  }

  window.onclick = (event) => {
    if (event.target == modal) {
      modal.style.display = "none"
      iframe.src = "" // Stop video
    }
  }
}

function showTrailerModal(performanceId) {
  const modal = document.getElementById("trailerModal")
  const iframe = document.getElementById("trailerIframe")

  // Idéalement, l'URL de la bande-annonce devrait venir de la base de données.
  // Pour l'exemple, on suppose une structure d'URL ou un endpoint API.
  // Si vous avez un champ trailer_url dans la table performances:
  // fetch(`api/trailer_url.php?id=${performanceId}`).then(res => res.json()).then(data => {
  //    if(data.success && data.url) iframe.src = data.url; else iframe.src = 'https://www.youtube.com/embed/dQw4w9WgXcQ'; // Fallback
  // });
  // Pour l'instant, on met un placeholder si api/trailer.php n'est pas prêt
  // En supposant que api/trailer.php renvoie l'URL embed de la vidéo
  fetch(`api/trailer.php?id=${performanceId}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success && data.trailer_url) {
        iframe.src = data.trailer_url // ex: https://www.youtube.com/embed/VIDEO_ID
        modal.style.display = "block"
      } else {
        showGlobalNotification(data.error || "Bande-annonce non disponible.", "info")
        //iframe.src = "https://www.youtube.com/embed/example"; // Fallback or error message
        //modal.style.display = "block";
      }
    })
    .catch((err) => {
      console.error("Error fetching trailer:", err)
      showGlobalNotification("Erreur lors du chargement de la bande-annonce.", "error")
      //iframe.src = "https://www.youtube.com/embed/example"; // Fallback
      //modal.style.display = "block";
    })
}

// S'assurer que showNotification est accessible globalement si utilisé par plusieurs scripts
// Si showNotification est dans main.js et que main.js est chargé, cela devrait fonctionner.
// Sinon, la définir ici ou l'importer si c'est un module.
function showGlobalNotification(message, type = "info") {
  if (typeof showNotification === "function") {
    // Tente d'utiliser la fonction showNotification de main.js
    showNotification(message, type)
  } else {
    // Fallback simple si showNotification n'est pas disponible
    alert(`${type.toUpperCase()}: ${message}`)
    console.log(`Notification (fallback): ${type.toUpperCase()}: ${message}`)
  }
}

function loadReviews() {
  const performanceId = new URLSearchParams(window.location.search).get("id")

  if (performanceId) {
    fetch(`api/reviews.php?performance_id=${performanceId}`)
      .then((response) => response.json())
      .then((data) => {
        displayReviews(data)
      })
      .catch((error) => {
        console.error("Erreur de chargement des avis:", error)
      })
  }
}

function displayReviews(reviews) {
  const reviewsContainer = document.getElementById("reviews-container")

  if (!reviewsContainer) return

  reviewsContainer.innerHTML = ""

  reviews.forEach((review) => {
    const reviewElement = createReviewElement(review)
    reviewsContainer.appendChild(reviewElement)
  })
}

function createReviewElement(review) {
  const reviewDiv = document.createElement("div")
  reviewDiv.className = "review-item"

  const stars = "★".repeat(review.rating) + "☆".repeat(5 - review.rating)

  reviewDiv.innerHTML = `
        <div class="review-header">
            <div class="review-author">${review.author_name}</div>
            <div class="review-rating">${stars}</div>
            <div class="review-date">${formatDate(review.created_at)}</div>
        </div>
        <div class="review-comment">${review.comment}</div>
    `

  return reviewDiv
}

function formatDate(dateString) {
  const date = new Date(dateString)
  return date.toLocaleDateString("fr-FR", {
    year: "numeric",
    month: "long",
    day: "numeric",
  })
}

function initImageCarousel() {
  const carousel = document.querySelector(".image-carousel")

  if (carousel) {
    let currentIndex = 0
    const images = carousel.querySelectorAll("img")
    const totalImages = images.length

    const prevButton = carousel.querySelector(".carousel-prev")
    const nextButton = carousel.querySelector(".carousel-next")

    if (prevButton) {
      prevButton.addEventListener("click", () => {
        currentIndex = (currentIndex - 1 + totalImages) % totalImages
        updateCarousel()
      })
    }

    if (nextButton) {
      nextButton.addEventListener("click", () => {
        currentIndex = (currentIndex + 1) % totalImages
        updateCarousel()
      })
    }

    function updateCarousel() {
      const offset = -currentIndex * 100
      carousel.querySelector(".carousel-track").style.transform = `translateX(${offset}%)`
    }
  }
}

// Mock functions to resolve the "noUndeclaredVariables" errors.
// In a real application, these would be defined elsewhere or imported.
function showNotification(message, type) {
  console.log(`Notification: ${message} (type: ${type})`)
}
