<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Non autorisé']);
    exit();
}

$database = new Database();
$db = $database->getConnection();
$user_id = $_SESSION['user_id'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['performance_id']) || !isset($input['event_date'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Données manquantes: performance_id et event_date requis.']);
        exit();
    }

    $performance_id = (int)$input['performance_id'];
    // Validate date format (YYYY-MM-DD HH:MM:SS)
    $event_date_str = $input['event_date'];
    $event_date_obj = DateTime::createFromFormat('Y-m-d H:i:s', $event_date_str);

    if (!$event_date_obj || $event_date_obj->format('Y-m-d H:i:s') !== $event_date_str) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Format de event_date invalide. Attendu: YYYY-MM-DD HH:MM:SS']);
        exit();
    }
    $event_date = $event_date_obj->format('Y-m-d H:i:s');

    try {
        // Vérifier si l'élément existe déjà
        $check_query = "SELECT 1 FROM watchlist 
                        WHERE user_id = :user_id 
                        AND performance_id = :performance_id 
                        AND event_date = :event_date";
        $check_stmt = $db->prepare($check_query);
        $check_stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
        $check_stmt->bindParam(':performance_id', $performance_id, PDO::PARAM_INT);
        $check_stmt->bindParam(':event_date', $event_date);
        $check_stmt->execute();

        if ($check_stmt->rowCount() > 0) {
            // Retirer de la watchlist
            $delete_query = "DELETE FROM watchlist 
                             WHERE user_id = :user_id 
                             AND performance_id = :performance_id 
                             AND event_date = :event_date";
            $delete_stmt = $db->prepare($delete_query);
            $delete_stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $delete_stmt->bindParam(':performance_id', $performance_id, PDO::PARAM_INT);
            $delete_stmt->bindParam(':event_date', $event_date);
            $delete_stmt->execute();
            echo json_encode(['success' => true, 'action' => 'removed', 'message' => 'Retiré de la liste "À venir".']);
        } else {
            // Ajouter à la watchlist
            $insert_query = "INSERT INTO watchlist (user_id, performance_id, event_date) 
                             VALUES (:user_id, :performance_id, :event_date)";
            $insert_stmt = $db->prepare($insert_query);
            $insert_stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $insert_stmt->bindParam(':performance_id', $performance_id, PDO::PARAM_INT);
            $insert_stmt->bindParam(':event_date', $event_date);
            $insert_stmt->execute();
            echo json_encode(['success' => true, 'action' => 'added', 'message' => 'Ajouté à la liste "À venir".']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        error_log("Watchlist API POST Error: " . $e->getMessage());
        echo json_encode(['success' => false, 'error' => 'Erreur serveur.', 'details' => $e->getMessage()]);
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        $query = "SELECT 
                        w.event_date,
                        p.id as performance_id, 
                        p.title as performance_title, 
                        p.poster_url,
                        p.slug as performance_slug,
                        c.name as category_name, 
                        AVG(r.rating) as avg_rating, 
                        COUNT(r.id) as review_count
                    FROM watchlist w
                    JOIN performances p ON w.performance_id = p.id
                    LEFT JOIN categories c ON p.category_id = c.id
                    LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                    WHERE w.user_id = :user_id
                    GROUP BY p.id, w.event_date, p.title, p.poster_url, p.slug, c.name
                    ORDER BY w.event_date ASC";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->execute();
        
        $watchlist_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'data' => $watchlist_items]);

    } catch (PDOException $e) {
        http_response_code(500);
        error_log("Watchlist API GET Error: " . $e->getMessage());
        echo json_encode(['success' => false, 'error' => 'Erreur serveur.', 'details' => $e->getMessage()]);
    }
} else {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['success' => false, 'error' => 'Méthode non autorisée.']);
}
?>
