<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Récupérer les spectacles à venir
$upcoming_query = "SELECT p.*, c.name as category_name, h.name as hall_name,
                   s.id as schedule_id, s.show_datetime, s.price, s.available_seats,
                   AVG(r.rating) as avg_rating, COUNT(DISTINCT r.id) as review_count,
                   f.user_id as is_favorite
                   FROM performances p
                   JOIN schedules s ON p.id = s.performance_id
                   JOIN halls h ON s.hall_id = h.id
                   LEFT JOIN categories c ON p.category_id = c.id
                   LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                   LEFT JOIN favorites f ON p.id = f.performance_id AND f.user_id = :user_id
                   WHERE s.show_datetime > NOW() AND s.status = 'open' AND p.status = 'online'
                   GROUP BY s.id
                   ORDER BY s.show_datetime ASC";
$upcoming_stmt = $db->prepare($upcoming_query);
$upcoming_stmt->bindParam(':user_id', $_SESSION['user_id']);
$upcoming_stmt->execute();
$upcoming_shows = $upcoming_stmt->fetchAll(PDO::FETCH_ASSOC);

// Grouper par date
$shows_by_date = [];
foreach ($upcoming_shows as $show) {
    $date = date('Y-m-d', strtotime($show['show_datetime']));
    $shows_by_date[$date][] = $show;
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>À Venir - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher un spectacle" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <div class="icon-btn notification-btn" title="Notifications"></div>
            </div>
        </header>

        <main class="main">
            <div class="page-header">
                <h2>📅 Spectacles à venir</h2>
                <p>Découvrez tous les spectacles programmés</p>
            </div>

            <!-- Filtres de période -->
            <div class="period-filters">
                <button class="period-btn active" data-period="all">Tous</button>
                <button class="period-btn" data-period="week">Cette semaine</button>
                <button class="period-btn" data-period="month">Ce mois</button>
                <button class="period-btn" data-period="next-month">Mois prochain</button>
            </div>

            <!-- Filtres par catégorie -->
            <div class="category-filters">
                <button class="category-filter active" data-category="all">Toutes les catégories</button>
                <button class="category-filter" data-category="comedie">Comédie</button>
                <button class="category-filter" data-category="drame">Drame</button>
                <button class="category-filter" data-category="musical">Musical</button>
                <button class="category-filter" data-category="classique">Classique</button>
            </div>

            <?php if (empty($upcoming_shows)): ?>
            <div class="empty-state">
                <div class="empty-icon">📅</div>
                <h3>Aucun spectacle programmé</h3>
                <p>Revenez bientôt pour découvrir notre nouvelle programmation.</p>
            </div>
            <?php else: ?>

            <!-- Timeline des spectacles -->
            <div class="upcoming-timeline">
                <?php foreach ($shows_by_date as $date => $shows): ?>
                <div class="timeline-date">
                    <div class="date-header">
                        <h3><?php echo strftime('%A %d %B %Y', strtotime($date)); ?></h3>
                        <span class="shows-count"><?php echo count($shows); ?> spectacle(s)</span>
                    </div>
                    
                    <div class="shows-list">
                        <?php foreach ($shows as $show): ?>
                        <div class="show-card" data-category="<?php echo strtolower($show['category_name']); ?>">
                            <div class="show-time">
                                <div class="time"><?php echo date('H:i', strtotime($show['show_datetime'])); ?></div>
                                <div class="hall"><?php echo htmlspecialchars($show['hall_name']); ?></div>
                            </div>
                            
                            <div class="show-poster">
                                <img src="<?php echo $show['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>" 
                                     alt="<?php echo htmlspecialchars($show['title']); ?>">
                            </div>
                            
                            <div class="show-info">
                                <div class="show-header">
                                    <h4><?php echo htmlspecialchars($show['title']); ?></h4>
                                    <span class="category-badge"><?php echo htmlspecialchars($show['category_name']); ?></span>
                                </div>
                                
                                <div class="show-details">
                                    <div class="rating">
                                        ★ <?php echo number_format($show['avg_rating'] ?: 0, 1); ?> 
                                        (<?php echo $show['review_count']; ?> avis)
                                    </div>
                                    <div class="duration"><?php echo $show['duration']; ?> min</div>
                                    <div class="price"><?php echo number_format($show['price'], 2); ?>€</div>
                                </div>
                                
                                <div class="availability">
                                    <span class="seats-left">
                                        <?php echo $show['available_seats']; ?> places disponibles
                                    </span>
                                    <?php if ($show['available_seats'] < 20): ?>
                                    <span class="urgency">Dernières places !</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="show-actions">
                                <button class="btn-favorite <?php echo $show['is_favorite'] ? 'active' : ''; ?>" 
                                        onclick="toggleFavorite(<?php echo $show['id']; ?>)">
                                    ♥
                                </button>
                                <a href="performance.php?id=<?php echo $show['id']; ?>" class="btn btn-outline">
                                    Détails
                                </a>
                                <a href="reservation.php?schedule_id=<?php echo $show['schedule_id']; ?>" class="btn btn-primary">
                                    Réserver
                                </a>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </main>

        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <div class="quick-book">
                <h4>⚡ Réservation rapide</h4>
                <p>Sélectionnez un spectacle pour réserver en un clic</p>
                <div class="selected-show" id="selectedShow" style="display: none;">
                    <div class="selected-info"></div>
                    <button class="btn btn-primary" id="quickBookBtn">Réserver maintenant</button>
                </div>
            </div>
            
            <div class="calendar-widget">
                <h4>📅 Ce mois-ci</h4>
                <div class="mini-calendar">
                    <!-- Mini calendrier généré par JS -->
                </div>
            </div>
        </aside>
    </div>

    <script src="assets/js/upcoming.js"></script>
</body>
</html>
