<?php
require_once 'config/database.php';

// Script de diagnostic pour vérifier le problème de connexion admin

$database = new Database();
$db = $database->getConnection();

$email = 'admin@nouvelacte.fr';
$password = 'admin123';

echo "<h2>🔍 Diagnostic de connexion admin</h2>";

try {
    // 1. Vérifier si l'utilisateur existe
    $query = "SELECT u.*, r.name as role_name FROM users u 
              JOIN roles r ON u.role_id = r.id 
              WHERE u.email = :email";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':email', $email);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo "✅ <strong>Utilisateur trouvé :</strong><br>";
        echo "📧 Email: {$user['email']}<br>";
        echo "👤 Nom: {$user['display_name']}<br>";
        echo "🔑 Rôle: {$user['role_name']}<br>";
        echo "🆔 ID: {$user['id']}<br>";
        echo "🔐 Hash actuel: " . substr($user['password_hash'], 0, 30) . "...<br><br>";
        
        // 2. Tester la vérification du mot de passe
        echo "<h3>🧪 Test de vérification du mot de passe :</h3>";
        
        if (password_verify($password, $user['password_hash'])) {
            echo "✅ <strong style='color: green;'>Mot de passe CORRECT</strong><br>";
            echo "Le problème ne vient pas du hash du mot de passe.<br>";
        } else {
            echo "❌ <strong style='color: red;'>Mot de passe INCORRECT</strong><br>";
            echo "Le hash dans la base de données ne correspond pas au mot de passe 'admin123'.<br><br>";
            
            // 3. Générer un nouveau hash correct
            $new_hash = password_hash($password, PASSWORD_DEFAULT);
            echo "<h3>🔧 Correction automatique :</h3>";
            
            $update_query = "UPDATE users SET password_hash = :new_hash WHERE email = :email";
            $update_stmt = $db->prepare($update_query);
            $update_stmt->bindParam(':new_hash', $new_hash);
            $update_stmt->bindParam(':email', $email);
            
            if ($update_stmt->execute()) {
                echo "✅ <strong style='color: green;'>Mot de passe corrigé avec succès !</strong><br>";
                echo "🔐 Nouveau hash: " . substr($new_hash, 0, 30) . "...<br>";
                
                // Vérifier que la correction fonctionne
                if (password_verify($password, $new_hash)) {
                    echo "✅ <strong style='color: green;'>Vérification du nouveau hash : SUCCÈS</strong><br>";
                } else {
                    echo "❌ <strong style='color: red;'>Erreur : Le nouveau hash ne fonctionne pas</strong><br>";
                }
            } else {
                echo "❌ <strong style='color: red;'>Erreur lors de la mise à jour</strong><br>";
            }
        }
        
    } else {
        echo "❌ <strong style='color: red;'>Utilisateur admin non trouvé !</strong><br>";
        echo "Création de l'utilisateur admin...<br><br>";
        
        // Créer l'utilisateur admin
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        $insert_query = "INSERT INTO users (email, password_hash, display_name, role_id) 
                        VALUES (:email, :password_hash, 'Administrateur', 1)";
        $insert_stmt = $db->prepare($insert_query);
        $insert_stmt->bindParam(':email', $email);
        $insert_stmt->bindParam(':password_hash', $password_hash);
        
        if ($insert_stmt->execute()) {
            echo "✅ <strong style='color: green;'>Utilisateur admin créé avec succès !</strong><br>";
        } else {
            echo "❌ <strong style='color: red;'>Erreur lors de la création</strong><br>";
        }
    }
    
    // 4. Test de simulation de connexion
    echo "<br><h3>🎭 Simulation de connexion :</h3>";
    
    $test_query = "SELECT u.*, r.name as role_name FROM users u 
                   JOIN roles r ON u.role_id = r.id 
                   WHERE u.email = :email";
    $test_stmt = $db->prepare($test_query);
    $test_stmt->bindParam(':email', $email);
    $test_stmt->execute();
    
    if ($test_stmt->rowCount() > 0) {
        $test_user = $test_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (password_verify($password, $test_user['password_hash'])) {
            echo "✅ <strong style='color: green;'>CONNEXION RÉUSSIE !</strong><br>";
            echo "🎉 Vous devriez maintenant pouvoir vous connecter avec :<br>";
            echo "📧 Email: {$email}<br>";
            echo "🔑 Mot de passe: {$password}<br>";
        } else {
            echo "❌ <strong style='color: red;'>CONNEXION ÉCHOUÉE</strong><br>";
            echo "Il y a encore un problème avec le mot de passe.<br>";
        }
    }
    
    // 5. Vérifier le code de identification.php
    echo "<br><h3>🔍 Vérification du code de connexion :</h3>";
    
    if (file_exists('identification.php')) {
        $login_content = file_get_contents('identification.php');
        
        if (strpos($login_content, 'password_verify') !== false) {
            echo "✅ Le fichier identification.php utilise password_verify()<br>";
        } else {
            echo "❌ Le fichier identification.php n'utilise pas password_verify()<br>";
            echo "⚠️ Vérifiez que le code utilise password_verify() et non une comparaison directe<br>";
        }
        
        if (strpos($login_content, 'JOIN roles') !== false) {
            echo "✅ Le fichier identification.php joint correctement la table roles<br>";
        } else {
            echo "⚠️ Vérifiez la requête SQL dans identification.php<br>";
        }
    } else {
        echo "❌ Fichier identification.php non trouvé<br>";
    }
    
} catch (PDOException $e) {
    echo "❌ Erreur de base de données: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug Login - Nouvel Acte</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 900px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
            line-height: 1.6;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h2, h3 {
            color: #e30b17;
            border-bottom: 2px solid #e30b17;
            padding-bottom: 5px;
        }
        .success {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
            border-left: 4px solid #28a745;
        }
        .error {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
            border-left: 4px solid #dc3545;
        }
        .actions {
            text-align: center;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #eee;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #e30b17;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin: 0 10px;
            font-weight: bold;
        }
        .btn:hover {
            background: #c5090f;
        }
        .btn-secondary {
            background: #6c757d;
        }
        code {
            background: #f8f9fa;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="success">
            <h3>🎯 Diagnostic terminé !</h3>
            <p>Le script a analysé et corrigé les problèmes de connexion. Essayez maintenant de vous connecter.</p>
        </div>
        
        <div class="actions">
            <a href="identification.php" class="btn">🔐 Tester la connexion</a>
            <a href="debug-login.php" class="btn btn-secondary">🔄 Relancer le diagnostic</a>
            <a href="index.html" class="btn btn-secondary">🏠 Accueil</a>
        </div>
    </div>
</body>
</html>
