<?php
session_start();
require_once 'config/database.php';

// Vérifier si l'utilisateur est connecté
if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Désactiver le mode strict pour cette session
$db->exec("SET sql_mode = ''");

// Récupérer les spectacles populaires
$query = "SELECT p.*, c.name as category_name, 
          AVG(r.rating) as avg_rating, COUNT(r.id) as review_count
          FROM performances p 
          LEFT JOIN categories c ON p.category_id = c.id
          LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
          WHERE p.status = 'online'
          GROUP BY p.id
          ORDER BY 
            CASE WHEN p.title = 'Hamlet' THEN 0 ELSE 1 END,
            avg_rating DESC, 
            review_count DESC
          LIMIT 10";
$stmt = $db->prepare($query);
$stmt->execute();
$popular_performances = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les catégories
$cat_query = "SELECT * FROM categories ORDER BY name";
$cat_stmt = $db->prepare($cat_query);
$cat_stmt->execute();
$categories = $cat_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les recommandations d'amis
$rec_query = "SELECT DISTINCT p.*, AVG(r.rating) as avg_rating, COUNT(r.id) as review_count
              FROM performances p
              JOIN reviews r ON p.id = r.performance_id
              JOIN friends f ON r.user_id = f.friend_id
              WHERE f.user_id = :user_id AND f.status = 'accepted' AND r.is_approved = 1
              GROUP BY p.id
              ORDER BY avg_rating DESC
              LIMIT 4";
$rec_stmt = $db->prepare($rec_query);
$rec_stmt->bindParam(':user_id', $_SESSION['user_id']);
$rec_stmt->execute();
$recommendations = $rec_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nouvel Acte Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Header -->
        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <div class="icon-btn notification-btn" title="Notifications"></div>
            </div>
        </header>

        <!-- Contenu principal -->
        <main class="main">
            <!-- Genres -->
            <div class="genres">
                <?php foreach ($categories as $category): ?>
                    <div class="genre" data-category="<?php echo $category['slug']; ?>">
                        <?php echo htmlspecialchars($category['name']); ?>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Spectacle en vedette -->
            <?php if (!empty($popular_performances)): ?>
                <?php $featured = $popular_performances[0]; ?>
                <section class="featured">
                    <div class="featured-bg" style="background-image: url('<?php echo $featured['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')"></div>
                    <div class="featured-content">
                        <h2><?php echo htmlspecialchars($featured['title']); ?></h2>
                        <div class="buttons">
                            <button class="btn-play" onclick="showTrailer(<?php echo $featured['id']; ?>)">Bande d'annonce</button>
                            <button class="btn-fav" onclick="toggleFavorite(<?php echo $featured['id']; ?>)"></button>
                        </div>
                    </div>
                </section>
            <?php endif; ?>

            <!-- Spectacles populaires -->
            <section class="popular">
                <div class="popular-header">
                    <h3>Spectacles Populaires</h3>
                    <a href="performances.php">Voir plus</a>
                </div>
                <div class="cards" id="popularCards">
                    <?php foreach (array_slice($popular_performances, 1, 5) as $performance): ?>
                        <div class="card-item" onclick="viewPerformance(<?php echo $performance['id']; ?>)">
                            <div class="card" style="background-image: url('<?php echo $performance['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')"></div>
                            <div class="card-info">
                                <div class="title"><?php echo htmlspecialchars($performance['title']); ?></div>
                                <div class="rating">
                                    ★ <?php echo number_format($performance['avg_rating'] ?: 0, 1); ?> 
                                    (<?php echo $performance['review_count']; ?>)
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>

            <!-- À venir -->
            <section class="popular">
                <div class="popular-header">
                    <h3>À Venir</h3>
                    <a href="upcoming.php">Voir plus</a>
                </div>
                <div class="cards">
                    <?php 
                    // Récupérer les prochaines séances
                    $upcoming_query = "SELECT p.*, 
                   AVG(r.rating) as avg_rating, 
                   COUNT(DISTINCT r.id) as review_count,
                   MIN(s.show_datetime) as next_show
                   FROM performances p
                   JOIN schedules s ON p.id = s.performance_id
                   LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                   WHERE s.show_datetime > NOW() AND s.status = 'open'
                   GROUP BY p.id, p.title, p.slug, p.description, p.duration, p.min_age, p.category_id, p.poster_url, p.status, p.created_at
                   ORDER BY next_show ASC
                   LIMIT 5";
                    $upcoming_stmt = $db->prepare($upcoming_query);
                    $upcoming_stmt->execute();
                    $upcoming_performances = $upcoming_stmt->fetchAll(PDO::FETCH_ASSOC);

                    foreach ($upcoming_performances as $performance): ?>
                        <div class="card-item" onclick="viewPerformance(<?php echo $performance['id']; ?>)">
                            <div class="card" style="background-image: url('<?php echo $performance['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')"></div>
                            <div class="card-info">
                                <div class="title"><?php echo htmlspecialchars($performance['title']); ?></div>
                                <div class="rating">
                                    ★ <?php echo number_format($performance['avg_rating'] ?: 0, 1); ?> 
                                    (<?php echo $performance['review_count']; ?>)
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>
        </main>

        <!-- Sidebar droite -->
        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <div class="recommendations">
                <p class="section-title">Aimé par tes amis</p>
                <div class="list">
                    <?php foreach ($recommendations as $rec): ?>
                        <div class="item" onclick="viewPerformance(<?php echo $rec['id']; ?>)">
                            <img src="<?php echo $rec['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>" alt=""/>
                            <div><?php echo htmlspecialchars($rec['title']); ?></div>
                            <div class="rating">
                                ★ <?php echo number_format($rec['avg_rating'], 1); ?> 
                                (<?php echo $rec['review_count']; ?>)
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <button class="btn-all" onclick="location.href='recommendations.php'">Voir tout</button>
        </aside>
    </div>

    <script src="assets/js/dashboard.js"></script>
</body>
</html>
