<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Récupérer toutes les catégories avec le nombre de spectacles
$categories_query = "SELECT c.*, COUNT(p.id) as performances_count,
                    AVG(r.rating) as avg_rating
                    FROM categories c
                    LEFT JOIN performances p ON c.id = p.category_id AND p.status = 'online'
                    LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                    GROUP BY c.id
                    ORDER BY c.name";
$categories_stmt = $db->prepare($categories_query);
$categories_stmt->execute();
$categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer la catégorie sélectionnée
$selected_category = $_GET['category'] ?? null;
$category_performances = [];

if ($selected_category) {
    $performances_query = "SELECT p.*, c.name as category_name,
                          AVG(r.rating) as avg_rating, COUNT(r.id) as review_count,
                          MIN(s.show_datetime) as next_show, MIN(s.price) as min_price,
                          f.user_id as is_favorite
                          FROM performances p
                          JOIN categories c ON p.category_id = c.id
                          LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                          LEFT JOIN schedules s ON p.id = s.performance_id AND s.show_datetime > NOW() AND s.status = 'open'
                          LEFT JOIN favorites f ON p.id = f.performance_id AND f.user_id = :user_id
                          WHERE c.slug = :category AND p.status = 'online'
                          GROUP BY p.id
                          ORDER BY avg_rating DESC, p.title";
    $performances_stmt = $db->prepare($performances_query);
    $performances_stmt->bindParam(':category', $selected_category);
    $performances_stmt->bindParam(':user_id', $_SESSION['user_id']);
    $performances_stmt->execute();
    $category_performances = $performances_stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Catégories - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher par catégorie" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <div class="icon-btn notification-btn" title="Notifications"></div>
            </div>
        </header>

        <main class="main">
            <?php if (!$selected_category): ?>
            <!-- Vue d'ensemble des catégories -->
            <div class="page-header">
                <h2>🎭 Catégories de spectacles</h2>
                <p>Explorez notre programmation par genre</p>
            </div>

            <div class="categories-grid">
                <?php foreach ($categories as $category): ?>
                <div class="category-card" onclick="location.href='?category=<?php echo $category['slug']; ?>'">
                    <div class="category-icon">
                        <?php
                        $icons = [
                            'comedie' => '😄',
                            'drame' => '🎭',
                            'musical' => '🎵',
                            'classique' => '🏛️',
                            'contemporain' => '🎨',
                            'jeune-public' => '👶'
                        ];
                        echo $icons[$category['slug']] ?? '🎪';
                        ?>
                    </div>
                    
                    <div class="category-content">
                        <h3><?php echo htmlspecialchars($category['name']); ?></h3>
                        <div class="category-stats">
                            <span class="performances-count">
                                <?php echo $category['performances_count']; ?> spectacle(s)
                            </span>
                            <?php if ($category['avg_rating']): ?>
                            <span class="avg-rating">
                                ★ <?php echo number_format($category['avg_rating'], 1); ?>
                            </span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="category-description">
                            <?php
                            $descriptions = [
                                'comedie' => 'Rires et bonne humeur garantis',
                                'drame' => 'Émotions intenses et réflexions profondes',
                                'musical' => 'Musique, chant et danse',
                                'classique' => 'Les grands classiques du théâtre',
                                'contemporain' => 'Créations modernes et audacieuses',
                                'jeune-public' => 'Spectacles adaptés aux enfants'
                            ];
                            echo $descriptions[$category['slug']] ?? 'Découvrez cette catégorie';
                            ?>
                        </div>
                    </div>
                    
                    <div class="category-arrow">→</div>
                </div>
                <?php endforeach; ?>
            </div>

            <?php else: ?>
            <!-- Vue détaillée d'une catégorie -->
            <?php
            $current_category = array_filter($categories, function($c) use ($selected_category) {
                return $c['slug'] === $selected_category;
            });
            $current_category = reset($current_category);
            ?>
            
            <div class="category-header">
                <button class="back-btn" onclick="location.href='categories.php'">← Retour</button>
                <div class="category-title">
                    <h2><?php echo htmlspecialchars($current_category['name']); ?></h2>
                    <p><?php echo $current_category['performances_count']; ?> spectacle(s) disponible(s)</p>
                </div>
            </div>

            <!-- Filtres -->
            <div class="category-filters">
                <button class="filter-btn active" data-sort="rating">Mieux notés</button>
                <button class="filter-btn" data-sort="title">Alphabétique</button>
                <button class="filter-btn" data-sort="recent">Plus récents</button>
                <button class="filter-btn" data-sort="available">Disponibles</button>
            </div>

            <?php if (empty($category_performances)): ?>
            <div class="empty-state">
                <div class="empty-icon">🎭</div>
                <h3>Aucun spectacle dans cette catégorie</h3>
                <p>Revenez bientôt pour découvrir notre nouvelle programmation.</p>
                <a href="categories.php" class="btn btn-primary">Voir toutes les catégories</a>
            </div>
            <?php else: ?>

            <div class="performances-grid">
                <?php foreach ($category_performances as $performance): ?>
                <div class="performance-card">
                    <div class="performance-image">
                        <img src="<?php echo $performance['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>" 
                             alt="<?php echo htmlspecialchars($performance['title']); ?>">
                        <button class="btn-favorite <?php echo $performance['is_favorite'] ? 'active' : ''; ?>" 
                                onclick="toggleFavorite(<?php echo $performance['id']; ?>)">
                            ♥
                        </button>
                        <?php if ($performance['next_show']): ?>
                        <div class="availability-badge">Disponible</div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="performance-content">
                        <h3><?php echo htmlspecialchars($performance['title']); ?></h3>
                        
                        <div class="performance-rating">
                            <div class="stars">
                                <?php 
                                $rating = round($performance['avg_rating'] ?: 0);
                                for ($i = 1; $i <= 5; $i++): ?>
                                    <span class="<?php echo $i <= $rating ? 'filled' : 'empty'; ?>">★</span>
                                <?php endfor; ?>
                            </div>
                            <span class="rating-text">
                                <?php echo number_format($performance['avg_rating'] ?: 0, 1); ?> 
                                (<?php echo $performance['review_count']; ?> avis)
                            </span>
                        </div>
                        
                        <div class="performance-details">
                            <div class="duration">⏱️ <?php echo $performance['duration']; ?> min</div>
                            <?php if ($performance['min_age'] > 0): ?>
                            <div class="age">👶 <?php echo $performance['min_age']; ?>+ ans</div>
                            <?php endif; ?>
                            <?php if ($performance['next_show']): ?>
                            <div class="next-show">
                                📅 <?php echo date('d/m/Y', strtotime($performance['next_show'])); ?>
                            </div>
                            <div class="price">
                                💰 À partir de <?php echo number_format($performance['min_price'], 2); ?>€
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="performance-description">
                            <?php echo htmlspecialchars(substr($performance['description'], 0, 120)) . '...'; ?>
                        </div>
                        
                        <div class="performance-actions">
                            <a href="performance.php?id=<?php echo $performance['id']; ?>" class="btn btn-outline">
                                Voir détails
                            </a>
                            <?php if ($performance['next_show']): ?>
                            <a href="reservation.php?performance_id=<?php echo $performance['id']; ?>" class="btn btn-primary">
                                Réserver
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
            <?php endif; ?>
        </main>

        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <?php if ($selected_category): ?>
            <div class="category-stats">
                <h4>📊 Statistiques</h4>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($category_performances); ?></span>
                    <span class="stat-label">Spectacles</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?php echo number_format($current_category['avg_rating'] ?: 0, 1); ?></span>
                    <span class="stat-label">Note moyenne</span>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="popular-categories">
                <h4>🔥 Catégories populaires</h4>
                <?php 
                $popular = array_slice(array_filter($categories, function($c) { return $c['performances_count'] > 0; }), 0, 3);
                foreach ($popular as $cat): ?>
                <div class="popular-item" onclick="location.href='?category=<?php echo $cat['slug']; ?>'">
                    <span class="cat-name"><?php echo htmlspecialchars($cat['name']); ?></span>
                    <span class="cat-count"><?php echo $cat['performances_count']; ?></span>
                </div>
                <?php endforeach; ?>
            </div>
        </aside>
    </div>

    <script src="assets/js/categories.js"></script>
</body>
</html>
