// JavaScript pour la page des spectacles à venir

document.addEventListener("DOMContentLoaded", () => {
  initUpcoming()
  initFilters()
  initQuickBook()
  initMiniCalendar()
})

function initUpcoming() {
  console.log("Page à venir initialisée")

  // Initialiser les interactions
  initShowCards()

  // Marquer les spectacles urgents
  markUrgentShows()
}

function initFilters() {
  // Filtres de période
  const periodButtons = document.querySelectorAll(".period-btn")
  periodButtons.forEach((button) => {
    button.addEventListener("click", function () {
      periodButtons.forEach((btn) => btn.classList.remove("active"))
      this.classList.add("active")

      const period = this.dataset.period
      filterByPeriod(period)
    })
  })

  // Filtres de catégorie
  const categoryButtons = document.querySelectorAll(".category-filter")
  categoryButtons.forEach((button) => {
    button.addEventListener("click", function () {
      categoryButtons.forEach((btn) => btn.classList.remove("active"))
      this.classList.add("active")

      const category = this.dataset.category
      filterByCategory(category)
    })
  })
}

function filterByPeriod(period) {
  const timelineDates = document.querySelectorAll(".timeline-date")
  const now = new Date()

  timelineDates.forEach((dateSection) => {
    const dateHeader = dateSection.querySelector(".date-header h3").textContent
    const showDate = new Date(dateHeader)
    let show = false

    switch (period) {
      case "all":
        show = true
        break
      case "week":
        const weekFromNow = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000)
        show = showDate <= weekFromNow
        break
      case "month":
        show = showDate.getMonth() === now.getMonth() && showDate.getFullYear() === now.getFullYear()
        break
      case "next-month":
        const nextMonth = new Date(now.getFullYear(), now.getMonth() + 1, 1)
        show = showDate.getMonth() === nextMonth.getMonth() && showDate.getFullYear() === nextMonth.getFullYear()
        break
    }

    dateSection.style.display = show ? "block" : "none"
  })
}

function filterByCategory(category) {
  const showCards = document.querySelectorAll(".show-card")

  showCards.forEach((card) => {
    if (category === "all" || card.dataset.category === category) {
      card.style.display = "flex"
    } else {
      card.style.display = "none"
    }
  })

  // Masquer les sections de date vides
  const timelineDates = document.querySelectorAll(".timeline-date")
  timelineDates.forEach((dateSection) => {
    const visibleShows = dateSection.querySelectorAll(".show-card:not([style*='display: none'])")
    dateSection.style.display = visibleShows.length > 0 ? "block" : "none"
  })
}

function initShowCards() {
  const showCards = document.querySelectorAll(".show-card")

  showCards.forEach((card) => {
    card.addEventListener("click", function (e) {
      if (!e.target.closest(".show-actions")) {
        selectShow(this)
      }
    })

    card.addEventListener("mouseenter", function () {
      this.style.transform = "translateX(5px)"
      this.style.boxShadow = "0 4px 12px rgba(0,0,0,0.15)"
    })

    card.addEventListener("mouseleave", function () {
      if (!this.classList.contains("selected")) {
        this.style.transform = "translateX(0)"
        this.style.boxShadow = ""
      }
    })
  })
}

function selectShow(card) {
  // Retirer la sélection précédente
  document.querySelectorAll(".show-card.selected").forEach((c) => {
    c.classList.remove("selected")
    c.style.transform = "translateX(0)"
    c.style.boxShadow = ""
  })

  // Sélectionner la nouvelle carte
  card.classList.add("selected")
  card.style.transform = "translateX(5px)"
  card.style.boxShadow = "0 4px 12px rgba(227, 11, 23, 0.3)"

  // Mettre à jour la réservation rapide
  updateQuickBook(card)
}

function initQuickBook() {
  const selectedShow = document.getElementById("selectedShow")
  const quickBookBtn = document.getElementById("quickBookBtn")

  if (quickBookBtn) {
    quickBookBtn.addEventListener("click", () => {
      const selectedCard = document.querySelector(".show-card.selected")
      if (selectedCard) {
        const reservationLink = selectedCard.querySelector("a[href*='reservation.php']")
        if (reservationLink) {
          window.location.href = reservationLink.href
        }
      }
    })
  }
}

function updateQuickBook(card) {
  const selectedShow = document.getElementById("selectedShow")
  const selectedInfo = selectedShow.querySelector(".selected-info")

  const title = card.querySelector("h4").textContent
  const time = card.querySelector(".time").textContent
  const price = card.querySelector(".price").textContent
  const hall = card.querySelector(".hall").textContent

  selectedInfo.innerHTML = `
    <h5>${title}</h5>
    <p>${time} - ${hall}</p>
    <p class="price">${price}</p>
  `

  selectedShow.style.display = "block"

  // Animation d'apparition
  selectedShow.style.opacity = "0"
  selectedShow.style.transform = "translateY(10px)"

  setTimeout(() => {
    selectedShow.style.transition = "all 0.3s ease"
    selectedShow.style.opacity = "1"
    selectedShow.style.transform = "translateY(0)"
  }, 100)
}

function markUrgentShows() {
  const urgencyElements = document.querySelectorAll(".urgency")

  urgencyElements.forEach((element) => {
    element.style.animation = "pulse 2s infinite"
  })
}

function initMiniCalendar() {
  const calendarContainer = document.querySelector(".mini-calendar")
  if (!calendarContainer) return

  const now = new Date()
  const year = now.getFullYear()
  const month = now.getMonth()

  // Créer le mini calendrier
  const calendar = createMiniCalendar(year, month)
  calendarContainer.innerHTML = calendar

  // Marquer les jours avec des spectacles
  markShowDays()
}

function createMiniCalendar(year, month) {
  const firstDay = new Date(year, month, 1)
  const lastDay = new Date(year, month + 1, 0)
  const daysInMonth = lastDay.getDate()
  const startingDayOfWeek = firstDay.getDay()

  const monthNames = [
    "Janvier",
    "Février",
    "Mars",
    "Avril",
    "Mai",
    "Juin",
    "Juillet",
    "Août",
    "Septembre",
    "Octobre",
    "Novembre",
    "Décembre",
  ]

  let calendar = `
    <div class="calendar-header">
      <h5>${monthNames[month]} ${year}</h5>
    </div>
    <div class="calendar-grid">
      <div class="day-header">D</div>
      <div class="day-header">L</div>
      <div class="day-header">M</div>
      <div class="day-header">M</div>
      <div class="day-header">J</div>
      <div class="day-header">V</div>
      <div class="day-header">S</div>
  `

  // Jours vides au début
  for (let i = 0; i < startingDayOfWeek; i++) {
    calendar += '<div class="calendar-day empty"></div>'
  }

  // Jours du mois
  for (let day = 1; day <= daysInMonth; day++) {
    const isToday = day === new Date().getDate() && month === new Date().getMonth()
    calendar += `<div class="calendar-day ${isToday ? "today" : ""}" data-date="${year}-${String(month + 1).padStart(2, "0")}-${String(day).padStart(2, "0")}">${day}</div>`
  }

  calendar += "</div>"
  return calendar
}

function markShowDays() {
  const timelineDates = document.querySelectorAll(".timeline-date")
  const calendarDays = document.querySelectorAll(".calendar-day[data-date]")

  const showDates = []
  timelineDates.forEach((dateSection) => {
    const dateHeader = dateSection.querySelector(".date-header h3").textContent
    const date = new Date(dateHeader)
    const dateString = date.toISOString().split("T")[0]
    showDates.push(dateString)
  })

  calendarDays.forEach((day) => {
    const dayDate = day.dataset.date
    if (showDates.includes(dayDate)) {
      day.classList.add("has-show")
      day.title = "Spectacles programmés"
    }
  })
}

function toggleFavorite(performanceId) {
  fetch("api/favorites.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      performance_id: performanceId,
      action: "toggle",
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        updateFavoriteButton(performanceId, data.is_favorite)
        showNotification(data.is_favorite ? "Ajouté aux favoris" : "Retiré des favoris", "success")
      } else {
        showNotification("Erreur lors de la mise à jour", "error")
      }
    })
    .catch((error) => {
      console.error("Erreur:", error)
      showNotification("Erreur de connexion", "error")
    })
}

function updateFavoriteButton(performanceId, isFavorite) {
  const buttons = document.querySelectorAll(`[onclick*="toggleFavorite(${performanceId})"]`)
  buttons.forEach((button) => {
    button.classList.toggle("active", isFavorite)
  })
}

function showNotification(message, type) {
  const notification = document.createElement("div")
  notification.className = `notification ${type}`
  notification.textContent = message

  document.body.appendChild(notification)

  setTimeout(() => {
    notification.classList.add("show")
  }, 100)

  setTimeout(() => {
    notification.classList.remove("show")
    setTimeout(() => notification.remove(), 300)
  }, 3000)
}
