// JavaScript pour la page des favoris

document.addEventListener("DOMContentLoaded", () => {
  initFavorites()
  initFilters()
  initSearch()
})

function initFavorites() {
  console.log("Page favoris initialisée")

  // Initialiser les interactions des cartes
  initCardInteractions()

  // Charger les animations
  animateCards()
}

function initFilters() {
  const filterButtons = document.querySelectorAll(".filter-btn")

  filterButtons.forEach((button) => {
    button.addEventListener("click", function () {
      // Retirer la classe active des autres boutons
      filterButtons.forEach((btn) => btn.classList.remove("active"))
      this.classList.add("active")

      const filter = this.dataset.filter
      filterFavorites(filter)
    })
  })
}

function filterFavorites(filter) {
  const cards = document.querySelectorAll(".favorite-card")

  cards.forEach((card) => {
    const hasNextShow = card.querySelector(".availability-badge.available")

    switch (filter) {
      case "all":
        card.style.display = "block"
        break
      case "available":
        card.style.display = hasNextShow ? "block" : "none"
        break
      case "upcoming":
        card.style.display = hasNextShow ? "block" : "none"
        break
    }
  })

  // Animer les cartes visibles
  setTimeout(() => {
    animateCards()
  }, 100)
}

function initSearch() {
  const searchInput = document.getElementById("searchInput")

  if (searchInput) {
    searchInput.addEventListener("input", function () {
      const query = this.value.toLowerCase()
      searchFavorites(query)
    })
  }
}

function searchFavorites(query) {
  const cards = document.querySelectorAll(".favorite-card")

  cards.forEach((card) => {
    const title = card.querySelector("h3").textContent.toLowerCase()
    const category = card.querySelector(".category").textContent.toLowerCase()

    if (title.includes(query) || category.includes(query)) {
      card.style.display = "block"
    } else {
      card.style.display = "none"
    }
  })
}

function initCardInteractions() {
  const cards = document.querySelectorAll(".favorite-card, .recommendation-card")

  cards.forEach((card) => {
    card.addEventListener("mouseenter", function () {
      this.style.transform = "translateY(-5px)"
    })

    card.addEventListener("mouseleave", function () {
      this.style.transform = "translateY(0)"
    })
  })
}

function animateCards() {
  const cards = document.querySelectorAll(".favorite-card:not([style*='display: none'])")

  cards.forEach((card, index) => {
    card.style.opacity = "0"
    card.style.transform = "translateY(20px)"

    setTimeout(() => {
      card.style.transition = "all 0.6s ease"
      card.style.opacity = "1"
      card.style.transform = "translateY(0)"
    }, index * 100)
  })
}

function toggleFavorite(performanceId) {
  fetch("api/favorites.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      performance_id: performanceId,
      action: "toggle",
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (!data.is_favorite) {
          // Retirer de la page des favoris avec animation
          const card = document.querySelector(`[onclick*="${performanceId}"]`).closest(".favorite-card")
          if (card) {
            card.style.transition = "all 0.3s ease"
            card.style.opacity = "0"
            card.style.transform = "scale(0.8)"

            setTimeout(() => {
              card.remove()
              updateFavoritesCount()
            }, 300)
          }
        }

        showNotification(data.is_favorite ? "Ajouté aux favoris" : "Retiré des favoris", "success")
      } else {
        showNotification("Erreur lors de la mise à jour", "error")
      }
    })
    .catch((error) => {
      console.error("Erreur:", error)
      showNotification("Erreur de connexion", "error")
    })
}

function updateFavoritesCount() {
  const remainingCards = document.querySelectorAll(".favorite-card").length
  const countElement = document.querySelector(".page-header p")

  if (countElement) {
    countElement.textContent = `${remainingCards} spectacle(s) dans vos favoris`
  }

  // Mettre à jour les statistiques
  const statNumber = document.querySelector(".stat-number")
  if (statNumber) {
    statNumber.textContent = remainingCards
  }

  // Afficher l'état vide si nécessaire
  if (remainingCards === 0) {
    showEmptyState()
  }
}

function showEmptyState() {
  const favoritesGrid = document.querySelector(".favorites-grid")
  const emptyState = `
    <div class="empty-state">
      <div class="empty-icon">💔</div>
      <h3>Plus de favoris</h3>
      <p>Vous avez retiré tous vos favoris. Découvrez de nouveaux spectacles !</p>
      <a href="dashboard.php" class="btn btn-primary">Découvrir des spectacles</a>
    </div>
  `

  favoritesGrid.innerHTML = emptyState
}

function viewPerformance(id) {
  window.location.href = `performance.php?id=${id}`
}

function showNotification(message, type) {
  const notification = document.createElement("div")
  notification.className = `notification ${type}`
  notification.textContent = message

  document.body.appendChild(notification)

  setTimeout(() => {
    notification.classList.add("show")
  }, 100)

  setTimeout(() => {
    notification.classList.remove("show")
    setTimeout(() => notification.remove(), 300)
  }, 3000)
}
