<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: ../identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$message = '';
$action = $_GET['action'] ?? 'list';
$performance_id = $_GET['id'] ?? 0;

// Traitement des actions
if ($_POST) {
    if ($action === 'add' || $action === 'edit') {
        $title = trim($_POST['title']);
        $description = trim($_POST['description']);
        $duration = (int)$_POST['duration'];
        $min_age = (int)$_POST['min_age'];
        $category_id = (int)$_POST['category_id'];
        $status = $_POST['status'];
        $slug = strtolower(str_replace(' ', '-', $title));
        
        if ($action === 'add') {
            $query = "INSERT INTO performances (title, slug, description, duration, min_age, category_id, status) 
                     VALUES (:title, :slug, :description, :duration, :min_age, :category_id, :status)";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':title', $title);
            $stmt->bindParam(':slug', $slug);
            $stmt->bindParam(':description', $description);
            $stmt->bindParam(':duration', $duration);
            $stmt->bindParam(':min_age', $min_age);
            $stmt->bindParam(':category_id', $category_id);
            $stmt->bindParam(':status', $status);
            
            if ($stmt->execute()) {
                $message = 'Spectacle ajouté avec succès';
                $action = 'list';
            }
        } else {
            $query = "UPDATE performances SET title = :title, slug = :slug, description = :description, 
                     duration = :duration, min_age = :min_age, category_id = :category_id, status = :status 
                     WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':title', $title);
            $stmt->bindParam(':slug', $slug);
            $stmt->bindParam(':description', $description);
            $stmt->bindParam(':duration', $duration);
            $stmt->bindParam(':min_age', $min_age);
            $stmt->bindParam(':category_id', $category_id);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':id', $performance_id);
            
            if ($stmt->execute()) {
                $message = 'Spectacle modifié avec succès';
                $action = 'list';
            }
        }
    }
}

// Suppression
if ($action === 'delete' && $performance_id) {
    $delete_query = "DELETE FROM performances WHERE id = :id";
    $delete_stmt = $db->prepare($delete_query);
    $delete_stmt->bindParam(':id', $performance_id);
    if ($delete_stmt->execute()) {
        $message = 'Spectacle supprimé avec succès';
    }
    $action = 'list';
}

// Récupération des données
if ($action === 'list') {
    $query = "SELECT p.*, c.name as category_name, 
              COUNT(DISTINCT s.id) as schedules_count,
              COUNT(DISTINCT r.id) as reviews_count,
              AVG(r.rating) as avg_rating
              FROM performances p 
              LEFT JOIN categories c ON p.category_id = c.id
              LEFT JOIN schedules s ON p.id = s.performance_id
              LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
              GROUP BY p.id
              ORDER BY p.created_at DESC";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $performances = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

if ($action === 'edit' && $performance_id) {
    $query = "SELECT * FROM performances WHERE id = :id";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $performance_id);
    $stmt->execute();
    $performance = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Récupération des catégories pour le formulaire
$categories_query = "SELECT * FROM categories ORDER BY name";
$categories_stmt = $db->prepare($categories_query);
$categories_stmt->execute();
$categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des spectacles - Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <header class="admin-header">
                <h1>Gestion des spectacles</h1>
                <?php if ($action === 'list'): ?>
                <a href="?action=add" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Ajouter un spectacle
                </a>
                <?php endif; ?>
            </header>

            <div class="admin-content">
                <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>

                <?php if ($action === 'list'): ?>
                <!-- Liste des spectacles -->
                <div class="admin-card">
                    <div class="table-responsive">
                        <table class="admin-table">
                            <thead>
                                <tr>
                                    <th>Titre</th>
                                    <th>Catégorie</th>
                                    <th>Durée</th>
                                    <th>Séances</th>
                                    <th>Note</th>
                                    <th>Statut</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($performances as $perf): ?>
                                <tr>
                                    <td>
                                        <div class="performance-info">
                                            <strong><?php echo htmlspecialchars($perf['title']); ?></strong>
                                            <small><?php echo htmlspecialchars(substr($perf['description'], 0, 50)) . '...'; ?></small>
                                        </div>
                                    </td>
                                    <td><?php echo htmlspecialchars($perf['category_name']); ?></td>
                                    <td><?php echo $perf['duration']; ?> min</td>
                                    <td><?php echo $perf['schedules_count']; ?></td>
                                    <td>
                                        <?php if ($perf['reviews_count'] > 0): ?>
                                            <span class="rating">
                                                ★ <?php echo number_format($perf['avg_rating'], 1); ?>
                                                (<?php echo $perf['reviews_count']; ?>)
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">Aucun avis</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="status status-<?php echo $perf['status']; ?>">
                                            <?php echo ucfirst($perf['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="?action=edit&id=<?php echo $perf['id']; ?>" class="btn btn-sm btn-outline">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <a href="schedules.php?performance_id=<?php echo $perf['id']; ?>" class="btn btn-sm btn-outline">
                                                <i class="fas fa-calendar"></i>
                                            </a>
                                            <a href="?action=delete&id=<?php echo $perf['id']; ?>" 
                                               class="btn btn-sm btn-danger" 
                                               onclick="return confirm('Êtes-vous sûr de vouloir supprimer ce spectacle ?')">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php elseif ($action === 'add' || $action === 'edit'): ?>
                <!-- Formulaire d'ajout/modification -->
                <div class="admin-card">
                    <div class="card-header">
                        <h3><?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?> un spectacle</h3>
                        <a href="?action=list" class="btn btn-outline">
                            <i class="fas fa-arrow-left"></i> Retour
                        </a>
                    </div>
                    
                    <form method="POST" class="admin-form">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="title">Titre *</label>
                                <input type="text" id="title" name="title" required 
                                       value="<?php echo htmlspecialchars($performance['title'] ?? ''); ?>">
                            </div>

                            <div class="form-group">
                                <label for="category_id">Catégorie *</label>
                                <select id="category_id" name="category_id" required>
                                    <option value="">Sélectionner une catégorie</option>
                                    <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>" 
                                            <?php echo ($performance['category_id'] ?? '') == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="form-group">
                                <label for="duration">Durée (minutes) *</label>
                                <input type="number" id="duration" name="duration" required min="1" 
                                       value="<?php echo htmlspecialchars($performance['duration'] ?? ''); ?>">
                            </div>

                            <div class="form-group">
                                <label for="min_age">Âge minimum</label>
                                <input type="number" id="min_age" name="min_age" min="0" max="18" 
                                       value="<?php echo htmlspecialchars($performance['min_age'] ?? '0'); ?>">
                            </div>

                            <div class="form-group">
                                <label for="status">Statut *</label>
                                <select id="status" name="status" required>
                                    <option value="draft" <?php echo ($performance['status'] ?? '') === 'draft' ? 'selected' : ''; ?>>Brouillon</option>
                                    <option value="online" <?php echo ($performance['status'] ?? '') === 'online' ? 'selected' : ''; ?>>En ligne</option>
                                    <option value="archived" <?php echo ($performance['status'] ?? '') === 'archived' ? 'selected' : ''; ?>>Archivé</option>
                                </select>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" rows="4"><?php echo htmlspecialchars($performance['description'] ?? ''); ?></textarea>
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i>
                                <?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?>
                            </button>
                            <a href="?action=list" class="btn btn-outline">Annuler</a>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <script src="assets/js/admin.js"></script>
</body>
</html>
