<?php
session_start();
require_once '../config/database.php';

// Vérifier si l'utilisateur est admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: ../identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Statistiques générales
$stats_query = "SELECT 
    (SELECT COUNT(*) FROM users WHERE role_id = 2) as total_users,
    (SELECT COUNT(*) FROM performances WHERE status = 'online') as total_performances,
    (SELECT COUNT(*) FROM reservations WHERE status = 'paid') as total_reservations,
    (SELECT SUM(total_price) FROM reservations WHERE status = 'paid') as total_revenue,
    (SELECT COUNT(*) FROM reviews WHERE is_approved = 0) as pending_reviews";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Réservations récentes
$recent_reservations_query = "SELECT r.*, u.display_name, u.email 
                             FROM reservations r 
                             JOIN users u ON r.user_id = u.id 
                             ORDER BY r.booked_at DESC 
                             LIMIT 5";
$recent_reservations_stmt = $db->prepare($recent_reservations_query);
$recent_reservations_stmt->execute();
$recent_reservations = $recent_reservations_stmt->fetchAll(PDO::FETCH_ASSOC);

// Performances populaires
$popular_performances_query = "SELECT p.title, COUNT(ri.id) as bookings, SUM(ri.price) as revenue
                              FROM performances p
                              JOIN schedules s ON p.id = s.performance_id
                              JOIN reservation_items ri ON s.id = ri.schedule_id
                              JOIN reservations r ON ri.reservation_id = r.id
                              WHERE r.status = 'paid'
                              GROUP BY p.id, p.title
                              ORDER BY bookings DESC
                              LIMIT 5";
$popular_performances_stmt = $db->prepare($popular_performances_query);
$popular_performances_stmt->execute();
$popular_performances = $popular_performances_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <header class="admin-header">
                <h1>Tableau de bord</h1>
                <div class="admin-user">
                    <span>Bienvenue, <?php echo htmlspecialchars($_SESSION['user_name']); ?></span>
                    <a href="../api/logout.php" class="btn-logout">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </header>

            <div class="admin-content">
                <!-- Statistiques principales -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-users"></i>
                        </div>
                        <div class="stat-content">
                            <h3><?php echo number_format($stats['total_users']); ?></h3>
                            <p>Utilisateurs</p>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-theater-masks"></i>
                        </div>
                        <div class="stat-content">
                            <h3><?php echo number_format($stats['total_performances']); ?></h3>
                            <p>Spectacles</p>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-ticket-alt"></i>
                        </div>
                        <div class="stat-content">
                            <h3><?php echo number_format($stats['total_reservations']); ?></h3>
                            <p>Réservations</p>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-euro-sign"></i>
                        </div>
                        <div class="stat-content">
                            <h3><?php echo number_format($stats['total_revenue'], 2); ?>€</h3>
                            <p>Chiffre d'affaires</p>
                        </div>
                    </div>
                </div>

                <!-- Alertes -->
                <?php if ($stats['pending_reviews'] > 0): ?>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <span><?php echo $stats['pending_reviews']; ?> avis en attente de modération</span>
                    <a href="reviews.php" class="btn btn-sm">Voir</a>
                </div>
                <?php endif; ?>

                <div class="dashboard-grid">
                    <!-- Réservations récentes -->
                    <div class="dashboard-card">
                        <div class="card-header">
                            <h3>Réservations récentes</h3>
                            <a href="reservations.php" class="btn btn-sm">Voir tout</a>
                        </div>
                        <div class="card-content">
                            <div class="table-responsive">
                                <table class="admin-table">
                                    <thead>
                                        <tr>
                                            <th>Client</th>
                                            <th>Montant</th>
                                            <th>Statut</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_reservations as $reservation): ?>
                                        <tr>
                                            <td>
                                                <div class="user-info">
                                                    <strong><?php echo htmlspecialchars($reservation['display_name']); ?></strong>
                                                    <small><?php echo htmlspecialchars($reservation['email']); ?></small>
                                                </div>
                                            </td>
                                            <td><?php echo number_format($reservation['total_price'], 2); ?>€</td>
                                            <td>
                                                <span class="status status-<?php echo $reservation['status']; ?>">
                                                    <?php echo ucfirst($reservation['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo date('d/m/Y H:i', strtotime($reservation['booked_at'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Spectacles populaires -->
                    <div class="dashboard-card">
                        <div class="card-header">
                            <h3>Spectacles populaires</h3>
                            <a href="performances.php" class="btn btn-sm">Gérer</a>
                        </div>
                        <div class="card-content">
                            <div class="popular-list">
                                <?php foreach ($popular_performances as $index => $performance): ?>
                                <div class="popular-item">
                                    <div class="popular-rank"><?php echo $index + 1; ?></div>
                                    <div class="popular-info">
                                        <strong><?php echo htmlspecialchars($performance['title']); ?></strong>
                                        <small><?php echo $performance['bookings']; ?> réservations - <?php echo number_format($performance['revenue'], 2); ?>€</small>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Actions rapides -->
                <div class="quick-actions">
                    <h3>Actions rapides</h3>
                    <div class="actions-grid">
                        <a href="performances.php?action=add" class="action-card">
                            <i class="fas fa-plus"></i>
                            <span>Ajouter un spectacle</span>
                        </a>
                        <a href="schedules.php?action=add" class="action-card">
                            <i class="fas fa-calendar-plus"></i>
                            <span>Programmer une séance</span>
                        </a>
                        <a href="users.php" class="action-card">
                            <i class="fas fa-user-plus"></i>
                            <span>Gérer les utilisateurs</span>
                        </a>
                        <a href="reviews.php" class="action-card">
                            <i class="fas fa-comments"></i>
                            <span>Modérer les avis</span>
                        </a>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script src="assets/js/admin.js"></script>
</body>
</html>
