// JavaScript pour l'administration

document.addEventListener("DOMContentLoaded", () => {
  // Initialisation
  initAdmin()
  initTables()
  initForms()
  initModals()
})

function initAdmin() {
  console.log("Admin panel initialized")

  // Gestion du menu mobile
  const menuToggle = document.querySelector(".menu-toggle")
  const sidebar = document.querySelector(".admin-sidebar")

  if (menuToggle && sidebar) {
    menuToggle.addEventListener("click", () => {
      sidebar.classList.toggle("open")
    })
  }

  // Auto-hide alerts
  const alerts = document.querySelectorAll(".alert")
  alerts.forEach((alert) => {
    setTimeout(() => {
      alert.style.opacity = "0"
      setTimeout(() => alert.remove(), 300)
    }, 5000)
  })
}

function initTables() {
  // Tri des tableaux
  const tables = document.querySelectorAll(".admin-table")

  tables.forEach((table) => {
    const headers = table.querySelectorAll("th[data-sort]")

    headers.forEach((header) => {
      header.style.cursor = "pointer"
      header.addEventListener("click", function () {
        sortTable(table, this.dataset.sort)
      })
    })
  })

  // Sélection multiple
  const selectAllCheckbox = document.querySelector("#select-all")
  const rowCheckboxes = document.querySelectorAll(".row-checkbox")

  if (selectAllCheckbox) {
    selectAllCheckbox.addEventListener("change", function () {
      rowCheckboxes.forEach((checkbox) => {
        checkbox.checked = this.checked
      })
      updateBulkActions()
    })
  }

  rowCheckboxes.forEach((checkbox) => {
    checkbox.addEventListener("change", updateBulkActions)
  })
}

function sortTable(table, column) {
  const tbody = table.querySelector("tbody")
  const rows = Array.from(tbody.querySelectorAll("tr"))
  const columnIndex = Array.from(table.querySelectorAll("th")).findIndex((th) => th.dataset.sort === column)

  rows.sort((a, b) => {
    const aValue = a.cells[columnIndex].textContent.trim()
    const bValue = b.cells[columnIndex].textContent.trim()

    if (!isNaN(aValue) && !isNaN(bValue)) {
      return Number.parseFloat(aValue) - Number.parseFloat(bValue)
    }

    return aValue.localeCompare(bValue)
  })

  rows.forEach((row) => tbody.appendChild(row))
}

function updateBulkActions() {
  const checkedBoxes = document.querySelectorAll(".row-checkbox:checked")
  const bulkActions = document.querySelector(".bulk-actions")

  if (bulkActions) {
    bulkActions.style.display = checkedBoxes.length > 0 ? "block" : "none"
  }
}

function initForms() {
  // Validation des formulaires
  const forms = document.querySelectorAll(".admin-form")

  forms.forEach((form) => {
    form.addEventListener("submit", function (e) {
      if (!validateForm(this)) {
        e.preventDefault()
      }
    })

    // Validation en temps réel
    const inputs = form.querySelectorAll("input, select, textarea")
    inputs.forEach((input) => {
      input.addEventListener("blur", function () {
        validateField(this)
      })
    })
  })

  // Auto-resize des textareas
  const textareas = document.querySelectorAll("textarea")
  textareas.forEach((textarea) => {
    textarea.addEventListener("input", function () {
      this.style.height = "auto"
      this.style.height = this.scrollHeight + "px"
    })
  })
}

function validateForm(form) {
  let isValid = true
  const requiredFields = form.querySelectorAll("[required]")

  requiredFields.forEach((field) => {
    if (!validateField(field)) {
      isValid = false
    }
  })

  return isValid
}

function validateField(field) {
  const value = field.value.trim()
  const isRequired = field.hasAttribute("required")
  let isValid = true

  // Supprimer les messages d'erreur existants
  const existingError = field.parentNode.querySelector(".field-error")
  if (existingError) {
    existingError.remove()
  }

  field.classList.remove("error")

  // Validation des champs requis
  if (isRequired && !value) {
    showFieldError(field, "Ce champ est requis")
    isValid = false
  }

  // Validation par type
  if (value) {
    switch (field.type) {
      case "email":
        if (!isValidEmail(value)) {
          showFieldError(field, "Email invalide")
          isValid = false
        }
        break
      case "number":
        if (isNaN(value)) {
          showFieldError(field, "Valeur numérique requise")
          isValid = false
        }
        break
    }
  }

  return isValid
}

function showFieldError(field, message) {
  field.classList.add("error")

  const errorDiv = document.createElement("div")
  errorDiv.className = "field-error"
  errorDiv.textContent = message

  field.parentNode.appendChild(errorDiv)
}

function isValidEmail(email) {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
  return emailRegex.test(email)
}

function initModals() {
  // Gestion des modales
  const modalTriggers = document.querySelectorAll("[data-modal]")
  const modals = document.querySelectorAll(".modal")

  modalTriggers.forEach((trigger) => {
    trigger.addEventListener("click", function (e) {
      e.preventDefault()
      const modalId = this.dataset.modal
      const modal = document.getElementById(modalId)
      if (modal) {
        showModal(modal)
      }
    })
  })

  modals.forEach((modal) => {
    const closeButtons = modal.querySelectorAll(".modal-close")
    closeButtons.forEach((button) => {
      button.addEventListener("click", () => hideModal(modal))
    })

    modal.addEventListener("click", function (e) {
      if (e.target === this) {
        hideModal(this)
      }
    })
  })
}

function showModal(modal) {
  modal.style.display = "flex"
  document.body.style.overflow = "hidden"

  setTimeout(() => {
    modal.classList.add("show")
  }, 10)
}

function hideModal(modal) {
  modal.classList.remove("show")

  setTimeout(() => {
    modal.style.display = "none"
    document.body.style.overflow = ""
  }, 300)
}

// Fonctions utilitaires
function showNotification(message, type = "success") {
  const notification = document.createElement("div")
  notification.className = `notification notification-${type}`
  notification.innerHTML = `
        <i class="fas fa-${type === "success" ? "check-circle" : "exclamation-triangle"}"></i>
        <span>${message}</span>
    `

  document.body.appendChild(notification)

  setTimeout(() => {
    notification.classList.add("show")
  }, 100)

  setTimeout(() => {
    notification.classList.remove("show")
    setTimeout(() => notification.remove(), 300)
  }, 3000)
}

function confirmAction(message, callback) {
  if (confirm(message)) {
    callback()
  }
}

// Actions AJAX
function deleteItem(url, itemId) {
  confirmAction("Êtes-vous sûr de vouloir supprimer cet élément ?", () => {
    fetch(url, {
      method: "DELETE",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({ id: itemId }),
    })
      .then((response) => response.json())
      .then((data) => {
        if (data.success) {
          showNotification("Élément supprimé avec succès")
          location.reload()
        } else {
          showNotification("Erreur lors de la suppression", "error")
        }
      })
      .catch((error) => {
        console.error("Error:", error)
        showNotification("Erreur de connexion", "error")
      })
  })
}

// Recherche en temps réel
function initSearch() {
  const searchInput = document.querySelector("#search")
  const searchResults = document.querySelector("#search-results")

  if (searchInput) {
    let searchTimeout

    searchInput.addEventListener("input", function () {
      clearTimeout(searchTimeout)
      const query = this.value.trim()

      if (query.length < 2) {
        if (searchResults) searchResults.innerHTML = ""
        return
      }

      searchTimeout = setTimeout(() => {
        performSearch(query)
      }, 300)
    })
  }
}

function performSearch(query) {
  fetch(`api/search.php?q=${encodeURIComponent(query)}`)
    .then((response) => response.json())
    .then((data) => {
      displaySearchResults(data)
    })
    .catch((error) => {
      console.error("Search error:", error)
    })
}

function displaySearchResults(results) {
  const searchResults = document.querySelector("#search-results")
  if (!searchResults) return

  if (results.length === 0) {
    searchResults.innerHTML = '<div class="no-results">Aucun résultat trouvé</div>'
    return
  }

  const html = results
    .map(
      (result) => `
        <div class="search-result">
            <h4>${result.title}</h4>
            <p>${result.description}</p>
            <a href="${result.url}">Voir</a>
        </div>
    `,
    )
    .join("")

  searchResults.innerHTML = html
}

// Export des fonctions globales
window.adminUtils = {
  showNotification,
  confirmAction,
  deleteItem,
  showModal,
  hideModal,
}
